#!/usr/bin/python3 -su

# Copyright (C) 2024 - 2025 ENCRYPTED SUPPORT LLC <adrelanos@kicksecure.com>
## See the file COPYING for copying conditions.

import sys
import subprocess
import os
import grp
from pathlib import Path
import tempfile

from PyQt5.QtWidgets import QApplication, QMainWindow, QDialog
from PyQt5.QtCore import Qt, pyqtSignal, QEvent, QTimer

## NOTE: _ui modules are autogenerated by build-ui.sh.
from sysmaint_panel.mainwindow_ui import Ui_MainWindow
from sysmaint_panel.reboot_ui import Ui_RebootDialog
from sysmaint_panel.shutdown_ui import Ui_ShutdownDialog
from sysmaint_panel.managesoftware_ui import Ui_ManageSoftwareDialog
from sysmaint_panel.managesoftwarehelp_ui import Ui_ManageSoftwareHelpDialog
from sysmaint_panel.background_ui import Ui_BackgroundScreen
from sysmaint_panel.nopriv_ui import Ui_NoPrivDialog
from sysmaint_panel.wronguser_ui import Ui_WrongUserDialog
from sysmaint_panel.uninstall_ui import Ui_UninstallDialog
from sysmaint_panel.managepasswords_ui import Ui_ManagePasswordsDialog
from sysmaint_panel.searchlogs_ui import Ui_SearchLogsDialog

# Honor sigterm
import signal

signal.signal(signal.SIGINT, signal.SIG_DFL)


def is_qubes_os():
    return Path("/usr/share/qubes/marker-vm").exists()


def is_kicksecure():
    return Path("/usr/share/kicksecure/marker").exists()


def is_whonix_gateway():
    return Path("/usr/share/anon-gw-base-files/gateway").exists()


def is_whonix_workstation():
    return Path("/usr/share/anon-ws-base-files/workstation").exists()


def timeout_lock(button):
    button_text_parts = button.text().split(" ")
    button_text_end_number = button_text_parts[
        len(button_text_parts) - 1
    ].strip("()")

    try:
        button_unlock_time = int(button_text_end_number)
    except Exception:
        button_text_parts.append("(5)")
        button.setText(" ".join(button_text_parts))
        button.setEnabled(False)
        QTimer.singleShot(1000, lambda: timeout_lock(button))
        return

    button_text_parts.pop()
    button_unlock_time -= 1
    if button_unlock_time == 0:
        button.setText(" ".join(button_text_parts))
        button.setEnabled(True)
        return

    button_text_parts.append(f"({button_unlock_time})")
    button.setText(" ".join(button_text_parts))
    QTimer.singleShot(1000, lambda: timeout_lock(button))


class NoPrivDialog(QDialog):
    def __init__(self):
        super(NoPrivDialog, self).__init__()
        self.ui = Ui_NoPrivDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.okButton.clicked.connect(self.done)


class WrongUserDialog(QDialog):
    def __init__(self):
        super(WrongUserDialog, self).__init__()
        self.ui = Ui_WrongUserDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.okButton.clicked.connect(self.done)


class BackgroundScreen(QDialog):
    def __init__(self):
        super(BackgroundScreen, self).__init__()
        self.ui = Ui_BackgroundScreen()
        self.ui.setupUi(self)
        if is_whonix_gateway():
            self.setStyleSheet("background-color: #77767b;")
        elif is_whonix_workstation():
            self.setStyleSheet("background-color: #4098bf;")
        elif is_kicksecure():
            self.setStyleSheet(
                "background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, "
                "y2:1, stop:0 #3b187b, stop:1 #a9def2);"
            )


class ManageSoftwareHelpDialog(QDialog):
    def __init__(self):
        super(ManageSoftwareHelpDialog, self).__init__()
        self.ui = Ui_ManageSoftwareHelpDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.okButton.clicked.connect(self.done)


class ManageSoftwareDialog(QDialog):
    def __init__(self):
        super(ManageSoftwareDialog, self).__init__()
        self.ui = Ui_ManageSoftwareDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.runButton.clicked.connect(self.run_action)
        self.ui.cancelButton.clicked.connect(self.cancel)

    def run_action(self):
        match self.ui.actionComboBox.currentText():
            case "Search":
                subprocess.Popen(
                    [
                        "/usr/libexec/helper-scripts/terminal-wrapper",
                        "/usr/bin/apt-cache",
                        "search",
                        self.ui.packageLineEdit.text(),
                    ]
                )
            case "Install":
                subprocess.Popen(
                    [
                        "/usr/libexec/helper-scripts/terminal-wrapper",
                        "/usr/bin/sudo",
                        "/usr/bin/apt",
                        "install",
                        self.ui.packageLineEdit.text(),
                    ]
                )
                self.done(0)
            case "Reinstall":
                subprocess.Popen(
                    [
                        "/usr/libexec/helper-scripts/terminal-wrapper",
                        "/usr/bin/sudo",
                        "/usr/bin/apt-get-reset",
                        self.ui.packageLineEdit.text(),
                    ]
                )
                self.done(0)
            case "Remove":
                subprocess.Popen(
                    [
                        "/usr/libexec/helper-scripts/terminal-wrapper",
                        "/usr/bin/sudo",
                        "/usr/bin/apt",
                        "remove",
                        self.ui.packageLineEdit.text(),
                    ]
                )
                self.done(0)
            case "Purge":
                subprocess.Popen(
                    [
                        "/usr/libexec/helper-scripts/terminal-wrapper",
                        "/usr/bin/sudo",
                        "/usr/bin/apt",
                        "purge",
                        self.ui.packageLineEdit.text(),
                    ]
                )
                self.done(0)
            case "Override":
                subprocess.Popen(
                    [
                        "/usr/libexec/helper-scripts/terminal-wrapper",
                        "/usr/bin/sudo",
                        "/usr/bin/dummy-dependency",
                        "--remove",
                        self.ui.packageLineEdit.text(),
                    ]
                )
                self.done(0)
            case "Help":
                help_window = ManageSoftwareHelpDialog()
                help_window.exec()

    def cancel(self):
        self.done(0)


class ManagePasswordsDialog(QDialog):
    def __init__(self):
        super(ManagePasswordsDialog, self).__init__()
        self.ui = Ui_ManagePasswordsDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.userPasswordButton.clicked.connect(self.user_password_change)
        self.ui.bootloaderPasswordButton.clicked.connect(
            self.bootloader_password_change
        )
        self.ui.diskPassphraseButton.clicked.connect(
            self.disk_passphrase_change
        )

    def user_password_change(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/sbin/pwchange",
            ]
        )
        self.done(0)

    def bootloader_password_change(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/sbin/grub-pwchange",
            ]
        )
        self.done(0)

    def disk_passphrase_change(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/sbin/crypt-pwchange",
            ]
        )
        self.done(0)


class RebootWindow(QDialog):
    def __init__(self):
        super(RebootWindow, self).__init__()
        self.ui = Ui_RebootDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.yesButton.clicked.connect(self.reboot)
        self.ui.noButton.clicked.connect(self.cancel)

    @staticmethod
    def reboot():
        subprocess.run(["/usr/sbin/reboot"])

    def cancel(self):
        self.done(0)


class ShutdownWindow(QDialog):
    def __init__(self):
        super(ShutdownWindow, self).__init__()
        self.ui = Ui_ShutdownDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.yesButton.clicked.connect(self.shutdown)
        self.ui.noButton.clicked.connect(self.cancel)

    @staticmethod
    def shutdown():
        subprocess.run(["/usr/sbin/shutdown", "now"])

    def cancel(self):
        self.done(0)


class UninstallDialog(QDialog):
    def __init__(self):
        super(UninstallDialog, self).__init__()
        self.ui = Ui_UninstallDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())
        self.ui.okButton.setEnabled(False)

        self.ui.okButton.clicked.connect(self.uninstall)
        self.ui.cancelButton.clicked.connect(self.cancel)
        self.ui.textField.textEdited.connect(self.check_text)

    closed = pyqtSignal()

    # Overrides QMainWindow.closeEvent
    def closeEvent(self, e):
        if xdg_current_desktop.startswith("sysmaint-session"):
            e.ignore()
            self.cancel()
        else:
            # noinspection PyUnresolvedReferences
            self.closed.emit()

    def check_text(self, text):
        if text == "yes":
            self.ui.okButton.setEnabled(True)
        else:
            self.ui.okButton.setEnabled(False)

    @staticmethod
    def uninstall():
        subprocess.run(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/bin/dummy-dependency",
                "--yes",
                "--purge",
                "user-sysmaint-split",
            ]
        )
        subprocess.run(["/usr/sbin/reboot"])

    @staticmethod
    def cancel():
        subprocess.run(["/usr/sbin/reboot"])


class SearchLogsDialog(QDialog):
    def __init__(self):
        super(SearchLogsDialog, self).__init__()
        self.ui = Ui_SearchLogsDialog()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())
        self.ui.searchButton.clicked.connect(self.search_logs)
        self.ui.cancelButton.clicked.connect(self.cancel)

    def search_logs(self):
        # We use pkexec here since we have to collect logs internally before
        # showing the terminal window, but also need to prompt for
        # authorization.
        journal_logs = subprocess.run(
            [
                "/usr/bin/pkexec",
                "/usr/bin/journalctl",
                "--no-pager",
                "--boot",
            ],
            capture_output=True,
            text=False,
        ).stdout
        filtered_logs = subprocess.run(
            [
                "/usr/bin/grep",
                "--extended-regexp",
                "--",
                self.ui.searchTermLineEdit.text(),
            ],
            capture_output=True,
            text=False,
            input=journal_logs,
        ).stdout
        temp_file = tempfile.NamedTemporaryFile(delete=False)
        temp_file.write(filtered_logs)
        temp_file.close()
        subprocess.run(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/less",
                temp_file.name,
            ]
        )
        os.unlink(temp_file.name)

    def cancel(self):
        self.done(0)


class MainWindow(QMainWindow):
    def __init__(self):
        super(MainWindow, self).__init__()
        self.ui = Ui_MainWindow()
        self.ui.setupUi(self)
        self.resize(self.minimumWidth(), self.minimumHeight())

        self.ui.installSystemButton.clicked.connect(self.install_system)

        self.ui.checkForUpdatesButton.clicked.connect(self.check_for_updates)
        self.ui.installUpdatesButton.clicked.connect(self.install_updates)
        self.ui.removeUnusedPackagesButton.clicked.connect(
            self.remove_unused_packages
        )
        self.ui.purgeUnusedPackagesButton.clicked.connect(
            self.purge_unused_packages
        )

        self.ui.browserInstallButton.clicked.connect(self.browser_install)
        self.ui.networkConnButton.clicked.connect(self.network_conn)

        self.ui.anonConnectionWizardButton.clicked.connect(self.anon_connection_wizard)
        self.ui.torControlPanelButton.clicked.connect(self.tor_control_panel)
        self.ui.onionCircuitsButton.clicked.connect(self.onion_circuits)
        self.ui.torStatusMonitorButton.clicked.connect(self.tor_status_monitor)

        self.ui.managePasswordsButton.clicked.connect(self.manage_passwords)
        self.ui.manageAutologinButton.clicked.connect(self.manage_autologin)
        self.ui.checkSystemStatusButton.clicked.connect(
            self.check_system_status
        )
        self.ui.createUserButton.clicked.connect(self.create_user)
        self.ui.removeUserButton.clicked.connect(self.remove_user)
        self.ui.runRepositoryWizardButton.clicked.connect(
            self.run_repository_wizard
        )
        self.ui.manageSoftwareButton.clicked.connect(self.manage_software)
        self.ui.searchLogsButton.clicked.connect(self.search_logs)

        self.ui.openTerminalButton.clicked.connect(self.open_terminal)
        self.ui.lockScreenButton.clicked.connect(self.lock_screen)
        self.ui.rebootButton.clicked.connect(self.reboot)
        self.ui.shutDownButton.clicked.connect(self.shutdown)

    closed = pyqtSignal()

    # Overrides QMainWindow.closeEvent
    def closeEvent(self, e):
        if (
            xdg_current_desktop.startswith("sysmaint-session")
            and not is_qubes_os()
        ):
            e.ignore()
            shutdown_window = ShutdownWindow()
            shutdown_window.exec()
        else:
            # noinspection PyUnresolvedReferences
            self.closed.emit()

    # Overrides QMainWindow.event
    def event(self, e):
        if (
            e.type() == QEvent.WindowStateChange
            and (self.windowState() & Qt.WindowMinimized) == Qt.WindowMinimized
        ):
            if xdg_current_desktop.startswith("sysmaint-session"):
                e.ignore()
                self.setWindowState(e.oldState())
                return True

        return super(MainWindow, self).event(e)

    def install_system(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/install-host",
            ]
        )
        timeout_lock(self.ui.installSystemButton)

    def check_for_updates(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/libexec/security-misc/apt-get-update",
            ]
        )
        timeout_lock(self.ui.checkForUpdatesButton)

    def install_updates(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/upgrade-nonroot",
            ]
        )
        timeout_lock(self.ui.installUpdatesButton)

    def remove_unused_packages(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/bin/apt",
                "autoremove",
            ]
        )
        timeout_lock(self.ui.removeUnusedPackagesButton)

    def purge_unused_packages(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/bin/apt",
                "autopurge",
            ]
        )
        timeout_lock(self.ui.purgeUnusedPackagesButton)

    @staticmethod
    def manage_software():
        manage_software_window = ManageSoftwareDialog()
        manage_software_window.exec()

    @staticmethod
    def search_logs():
        search_logs_window = SearchLogsDialog()
        search_logs_window.exec()

    def browser_install(self):
        subprocess.Popen(["/usr/bin/browser-choice"])
        timeout_lock(self.ui.browserInstallButton)

    def network_conn(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/nmtui",
            ]
        )
        timeout_lock(self.ui.networkConnButton)

    def anon_connection_wizard(self):
        subprocess.Popen(["/usr/bin/anon-connection-wizard"])
        timeout_lock(self.ui.anonConnectionWizardButton)

    def tor_control_panel(self):
        subprocess.Popen(["/usr/bin/tor-control-panel"])
        timeout_lock(self.ui.torControlPanelButton)

    def onion_circuits(self):
        subprocess.Popen(["/usr/bin/onioncircuits"])
        timeout_lock(self.ui.onionCircuitsButton)

    def tor_status_monitor(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/nyx",
            ]
        )
        timeout_lock(self.ui.torStatusMonitorButton)

    @staticmethod
    def manage_passwords():
        manage_passwords_window = ManagePasswordsDialog()
        manage_passwords_window.exec()

    def create_user(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/libexec/sysmaint-panel/create-user",
            ]
        )
        timeout_lock(self.ui.createUserButton)

    def remove_user(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/sbin/deluser",
            ]
        )
        timeout_lock(self.ui.removeUserButton)

    def run_repository_wizard(self):
        subprocess.Popen(
            ["/usr/libexec/repository-dist/repository-dist-wizard"]
        )
        timeout_lock(self.ui.runRepositoryWizardButton)

    def manage_autologin(self):
        subprocess.Popen(
            [
                "/usr/libexec/helper-scripts/terminal-wrapper",
                "/usr/bin/sudo",
                "/usr/sbin/autologinchange",
            ]
        )
        timeout_lock(self.ui.manageAutologinButton)

    def check_system_status(self):
        subprocess.Popen(["/usr/bin/systemcheck", "--gui"])
        timeout_lock(self.ui.checkSystemStatusButton)

    @staticmethod
    def open_terminal():
        subprocess.Popen(
            ["/usr/libexec/helper-scripts/terminal-wrapper", default_shell]
        )

    def lock_screen(self):
        subprocess.Popen(["/usr/libexec/helper-scripts/lock-screen"])
        timeout_lock(self.ui.lockScreenButton)

    @staticmethod
    def reboot():
        reboot_window = RebootWindow()
        reboot_window.exec()

    @staticmethod
    def shutdown():
        shutdown_window = ShutdownWindow()
        shutdown_window.exec()


def main():
    app = QApplication(sys.argv)

    sudo_stat_info = os.stat("/usr/bin/sudo")
    sudo_owning_gid = sudo_stat_info.st_gid
    sudo_owning_group = grp.getgrgid(sudo_owning_gid)[0]
    if sudo_owning_group == "sysmaint":
        if not os.access("/usr/bin/sudo", os.X_OK):
            if "boot-role=sysmaint" in kernel_cmdline:
                wuwin = WrongUserDialog()
                wuwin.show()
            else:
                npwin = NoPrivDialog()
                npwin.show()
            sys.exit(app.exec_())

    if "remove-sysmaint" in kernel_cmdline:
        window = UninstallDialog()
    else:
        window = MainWindow()
        if (
            not "rd.live.image" in kernel_cmdline
            and not "remove-sysmaint" in kernel_cmdline
        ):
            window.ui.installationGroupBox.setVisible(False)
        if is_qubes_os():
            window.ui.rebootButton.setVisible(False)
        if is_whonix_gateway() or is_whonix_workstation():
            window.ui.browserInstallButton.setVisible(False)
            window.ui.networkConnButton.setVisible(False)
        if not is_whonix_gateway():
            window.ui.anonConnectionWizardButton.setVisible(False)
            window.ui.torControlPanelButton.setVisible(False)
            window.ui.onionCircuitsButton.setVisible(False)
            window.ui.torStatusMonitorButton.setVisible(False)

    window.show()

    if xdg_current_desktop.startswith("sysmaint-session") and not is_qubes_os():
        bgrd_list = []
        for screen in app.screens():
            bgrd = BackgroundScreen()
            bgrd.setGeometry(screen.geometry())
            bgrd.setWindowFlags(
                Qt.WindowStaysOnBottomHint | Qt.WindowDoesNotAcceptFocus
            )
            bgrd.showFullScreen()
            # noinspection PyUnresolvedReferences
            window.closed.connect(bgrd.close)
            bgrd_list.append(bgrd)

    sys.exit(app.exec_())


xdg_current_desktop = ""
if "XDG_CURRENT_DESKTOP" in os.environ:
    xdg_current_desktop = os.environ["XDG_CURRENT_DESKTOP"]
default_shell = "/bin/bash"
if "SHELL" in os.environ:
    default_shell = os.environ["SHELL"]

with open("/proc/cmdline", "r") as kernel_cmdline_file:
    kernel_cmdline = kernel_cmdline_file.read()

if __name__ == "__main__":
    main()
