# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

from pathlib import Path
from typing import Dict, Tuple

from mpp import (
    EmonParser,
    EmonSystemInformationParser,
    JsonConstantParser,
    PcmParser,
    PcmSystemInformationParser,
    PcmSymbolTable, DataParser
)
from mpp.parsers.system_information_parser import SymbolTable, SystemInformationParser


class DataParserFactory:
    """
    Create an event collector parser based on collector detected in event file
    """
    parsers_for_collector = {
        'EMON': EmonParser,
        'PCM': PcmParser,
    }

    @classmethod
    def create(cls, input_data_file_path, frequency=None) -> DataParser:
        """
        Creates a parser object based on the given collector detected from event file
        :param input_data_file_path: emon_dat_file_path to parse
        :param frequency: optional frequency
        :return: an appropriate event collector parser suitable for the specified file
        """
        input_file = Path(input_data_file_path)
        ref_freq = cls.get_frequency_in_hz(frequency)
        num_of_lines_to_check = 10
        with open(input_file, 'r') as f:
            line_enumerator = enumerate(f)
            collector = cls.__skip_lines_until_collector_mention(line_enumerator, num_of_lines_to_check)

        if not collector:
            raise ValueError(f'Invalid input file {input_file}: unrecognized data format')
        return cls.parsers_for_collector[collector](input_file, timezone=None, ref_tsc_hz=ref_freq)

    @staticmethod
    def get_frequency_in_hz(ref_freq_mhz):
        return ref_freq_mhz * 1000000 if ref_freq_mhz else 0

    @staticmethod
    def __skip_lines_until_collector_mention(lines, n):
        for index, line in lines:
            line = line.strip()
            if line and line.startswith('EMON Version'):
                return 'EMON'
            if line and (line.startswith('Processor Counter Monitor:') or line.startswith(
                    'Intel(r) Performance Counter Monitor:')):
                return 'PCM'
            if index == n:
                return None
        return None


class SymbolTableFactory:
    """
    Yield a symbol table based on collector detected in event file
    """
    parsers_for_collector = {
        EmonSystemInformationParser: SymbolTable,
        PcmSystemInformationParser: PcmSymbolTable,
        # PerfSystemInformationParser : PerfSystemINformationAdapter
    }

    @classmethod
    def create(cls, system_info, core_type, latency_file_map):
        symbol_table = cls._get_symbols(system_info, core_type)
        symbol_table.update(cls._parse_latency_symbols(latency_file_map, core_type))
        return symbol_table

    @classmethod
    def _get_symbols(cls, system_info: SystemInformationParser, core_type):
        if type(system_info) not in cls.parsers_for_collector:
            raise ValueError(f'No System Information Parser defined for {type(system_info)}')
        return cls.parsers_for_collector[type(system_info)](system_info.processor_features,
                                                            system_info.system_features,
                                                            system_info.uncore_units, system_info.ref_tsc,
                                                            system_info.unique_core_types,
                                                            system_info.parser_attributes.symbols)\
            .get_symbol_table(core_type)

    @classmethod
    def _parse_latency_symbols(cls, latency_file_map, core_type, latency_descriptor: str = 'MEAN') -> Dict[str, int]:
        if latency_file_map and core_type in latency_file_map.keys():
            json_constant_parser = JsonConstantParser(latency_file_map[core_type], latency_descriptor)
            return json_constant_parser.parse()
        return {}
