/*******************************************************************************
* Copyright 2024 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP sycl::buffer API oneapi::mkl::sparse::omatadd to perform general
*       sparse matrix-sparse matrix addition on a SYCL device (CPU, GPU).
*       This example uses matrices in CSR format.
*
*           C = alpha * op(A) + beta * op(B)
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       The supported floating point data types for omatadd matrix data are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix format for omatadd is:
*           CSR + CSR = CSR
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix-Sparse Matrix Addition consisting of
// initialization of A and B matrices through process of creating C matrix as
// the sum,
//
// C = alpha * op(A) + beta * op(B)
//
template <typename fpType, typename intType>
int run_sparse_matrix_sparse_matrix_add_example(const sycl::device &dev)
{
    // Initialize data for Sparse Matrix - Sparse Matrix Addition
    oneapi::mkl::transpose opA = oneapi::mkl::transpose::nontrans;
    oneapi::mkl::transpose opB = oneapi::mkl::transpose::nontrans;

    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::zero;
    oneapi::mkl::index_base b_index = oneapi::mkl::index_base::zero;
    oneapi::mkl::index_base c_index = oneapi::mkl::index_base::zero;

    fpType alpha = set_fp_value(fpType(2.0), fpType(0.0));
    fpType beta  = set_fp_value(fpType(1.0), fpType(0.0));

    //
    // set up dimensions of matrix addition
    //
    intType size = 4;

    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;
    intType b_nrows = (opA == opB) ? a_nrows : a_ncols;
    intType b_ncols = (opA == opB) ? a_ncols : a_nrows;
    intType c_nrows = (opA == oneapi::mkl::transpose::nontrans) ? a_nrows : a_ncols;
    intType c_ncols = (opA == oneapi::mkl::transpose::nontrans) ? a_ncols : a_nrows;
    // c_nnz is unknown at this point

    //
    // setup A data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fpType, mkl_allocator<fpType, 64>> a;

    ia.resize(a_nrows + 1);
    ja.resize(27 * a_nrows);
    a.resize(27 * a_nrows);

    intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fpType, intType>(size, ia, ja, a, a_ind);
    a_nnz = ia[a_nrows] - a_ind;

    //
    // setup B data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ib;
    std::vector<intType, mkl_allocator<intType, 64>> jb;
    std::vector<fpType, mkl_allocator<fpType, 64>> b;

    intType b_ind = b_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_random_sparse_matrix<fpType, intType>(b_nrows, b_ncols, 0.1 /*density_val*/, ib, jb, b, b_ind);
    intType b_nnz = ib[b_nrows] - b_ind;

    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::omatadd:\n"
                          << e.what() << std::endl;
            }
        }
    };

    //
    // Execute Matrix Addition
    //

    std::cout << "\n\t\tsparse::omatadd parameters:\n";
    std::cout << "\t\t\topA = " << opA << std::endl;
    std::cout << "\t\t\topB = " << opB << std::endl;

    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
    std::cout << "\t\t\tB_nrows = B_ncols = " << b_nrows << std::endl;
    std::cout << "\t\t\tC_nrows = C_ncols = " << c_nrows << std::endl;

    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tB_index = " << b_index << std::endl;
    std::cout << "\t\t\tC_index = " << c_index << std::endl;

    oneapi::mkl::sparse::matrix_handle_t A = nullptr;
    oneapi::mkl::sparse::matrix_handle_t B = nullptr;
    oneapi::mkl::sparse::matrix_handle_t C = nullptr;

    oneapi::mkl::sparse::omatadd_descr_t descr = nullptr;

    sycl::buffer<std::uint8_t> *tempWorkspace = nullptr;

    // create execution main_queue and buffers of matrix data
    sycl::queue main_queue(dev, exception_handler);

    sycl::buffer<intType, 1> a_rowptr(ia.data(), a_nrows + 1);
    sycl::buffer<intType, 1> a_colind(ja.data(), a_nnz);
    sycl::buffer<fpType, 1> a_val(a.data(), a_nnz);
    sycl::buffer<intType, 1> b_rowptr(ib.data(), b_nrows + 1);
    sycl::buffer<intType, 1> b_colind(jb.data(), b_nnz);
    sycl::buffer<fpType, 1> b_val(b.data(), b_nnz);

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);
        oneapi::mkl::sparse::init_matrix_handle(&B);
        oneapi::mkl::sparse::init_matrix_handle(&C);

        oneapi::mkl::sparse::set_csr_data(main_queue, A, a_nrows, a_ncols, a_index, a_rowptr, a_colind, a_val);
        oneapi::mkl::sparse::set_csr_data(main_queue, B, b_nrows, b_ncols, b_index, b_rowptr, b_colind, b_val);

        // Step 1.1: Create 0-sized dummy args for now to matrix handle
        sycl::buffer<intType, 1> c_rowptr_dummy(0);
        sycl::buffer<intType, 1> c_colind_dummy(0);
        sycl::buffer<fpType, 1> c_vals_dummy(0);
        oneapi::mkl::sparse::set_csr_data(main_queue, C, c_nrows, c_ncols, c_index,
                                          c_rowptr_dummy, c_colind_dummy, c_vals_dummy);

        // Step 1.2: Initialize the omatadd descriptor and set algorithm
        oneapi::mkl::sparse::omatadd_alg alg = oneapi::mkl::sparse::omatadd_alg::default_alg;
        oneapi::mkl::sparse::init_omatadd_descr(main_queue, &descr);

        // Step 2.1: Query for size of temporary workspace
        std::int64_t sizeTempWorkspace = 0;
        oneapi::mkl::sparse::omatadd_buffer_size(main_queue, opA, opB, A, B, C, alg, descr, sizeTempWorkspace);

        // Step 2.2: Allocate temporary workspace
        tempWorkspace = new sycl::buffer<std::uint8_t, 1>(sizeTempWorkspace);

        // Step 3.1: Analyze sparity patterns of A and B to count non-zeros in C
        oneapi::mkl::sparse::omatadd_analyze(main_queue, opA, opB, A, B, C, alg, descr, tempWorkspace);

        // Step 3.2: Get non-zero count of output C matrix
        std::int64_t c_nnz = 0;
        oneapi::mkl::sparse::omatadd_get_nnz(main_queue, opA, opB, A, B, C, alg, descr, c_nnz);

        // Step 4.1: Allocate final c matrix arrays
        sycl::buffer<intType, 1> c_rowptr(c_nrows + 1);
        sycl::buffer<intType, 1> c_colind(c_nnz);
        sycl::buffer<fpType, 1> c_vals(c_nnz);
        oneapi::mkl::sparse::set_csr_data(main_queue, C, c_nrows, c_ncols, c_index, c_rowptr, c_colind, c_vals);

        // Step 4.2: Finalize by performing matrix addition into C matrix handle
        oneapi::mkl::sparse::omatadd(main_queue, opA, opB, alpha, A, beta, B, C, alg, descr);

        // Step 4.3: Release omatadd descriptor and temporary workspace
        oneapi::mkl::sparse::release_omatadd_descr(main_queue, descr);
        delete tempWorkspace; tempWorkspace = nullptr;

        // Step 5 (Optional): Sort C matrix output if desired
        oneapi::mkl::sparse::sort_matrix(main_queue, C);

        // Print part of C matrix solution
        {
            auto ic       = c_rowptr.get_host_access(sycl::read_only);
            auto jc       = c_colind.get_host_access(sycl::read_only);
            auto c        = c_vals.get_host_access(sycl::read_only);
            intType c_ind = c_index == oneapi::mkl::index_base::zero ? 0 : 1;
            std::cout << "C matrix [first two rows]:" << std::endl;
            for (intType row = 0; row < std::min(static_cast<intType>(2), c_nrows); ++row) {
                for (intType j = ic[row] - c_ind; j < ic[row + 1] - c_ind; ++j) {
                    intType col = jc[j];
                    fpType val  = c[j];
                    std::cout << "C(" << row + c_ind << ", " << col << ") = " << val << std::endl;
                }
            }
        }

        // Clean up
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &C);

    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_omatadd_descr(main_queue, descr);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &C);

        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_omatadd_descr(main_queue, descr);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B);
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &C);

        return 1;
    }

    main_queue.wait();

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Sparse Matrix Addition Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#    C = alpha * op(A) + beta * op(B)" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A and B are sparse matrices in CSR format, and C is the\n"
                 "# output of sparse matrix addition in CSR format"
              << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::omatadd_buffer_size" << std::endl;
    std::cout << "#   sparse::omatadd_analyze" << std::endl;
    std::cout << "#   sparse::omatadd_get_nnz" << std::endl;
    std::cout << "#   sparse::omatadd" << std::endl;
    std::cout << "#   sparse::init_omatadd_descr" << std::endl;
    std::cout << "#   sparse::release_omatadd_descr" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixAdditionExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_sparse_matrix_add_example<float, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_sparse_matrix_add_example<double, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }

            std::cout << "\tRunning with single precision complex data type:" << std::endl;
            status = run_sparse_matrix_sparse_matrix_add_example<std::complex<float>, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision complex data type:" << std::endl;
                status = run_sparse_matrix_sparse_matrix_add_example<std::complex<double>, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    mkl_free_buffers();
    return status;
}
