!=======================================================================
!
!  SAMPLE SOURCE CODE - SUBJECT TO THE TERMS OF END-USER LICENSE AGREEMENT FOR
!  INTEL(R) ADVISOR XE 2013.
!
!  Copyright (C) 2013 Intel Corporation. All rights reserved
!
!  THIS FILE IS PROVIDED "AS IS" WITH NO WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT
!  NOT LIMITED TO ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
!  PURPOSE, NON-INFRINGEMENT OF INTELLECTUAL PROPERTY RIGHTS.
!
! ========================================================================

program MatrixMultiplication

!The Advisor XE annotation library
use advisor_annotate

implicit none

integer :: size = 0

character(10) :: cmdarg
integer :: stat
integer :: time_start, time_end, count_rate

real, allocatable, dimension(:,:)  :: aMatrix, bMatrix, product

! Get command line argument
if (command_argument_count() < 1) then
  print *, "Usage: 2_mmult_annotated[_debug] arraySize"
  size = 1024
  print *, "Using default size of 1024"
else
  call get_command_argument (1, cmdarg, status=stat)
  if (stat /= 0) then
    print *, "Invalid arraySize, it should be between 2 and 9000"
    stop
  end if
  read (cmdarg,*,iostat=stat) size
  if ((stat /= 0) .or. (size < 2) .or. (size > 9000)) then
    print *, "Error: arraySize must be between 2 and 9000"
    stop
  end if
endif

! allocate matrices
allocate ( aMatrix(size,size), bMatrix(size,size), product(size,size) )

! initialize the arrays with different data
call init_array(3.0, -2.0, 1.0, size, aMatrix)
call init_array(-2.0, 1.0, 3.0, size, bMatrix)

print 101, "Matrix size: ", size, " X ", size
101 format (A,I0,A,I0,A)

! start timing the matrix multiply code
call system_clock (time_start)
call matrix_multiply (size, aMatrix, bMatrix, product)
! stop timing the matrix multiply code
call system_clock (time_end, count_rate)

! print simple test case of data to be sure multiplication is correct
if (size < 6) then
  call print_array("aMatrix", size, aMatrix)
  call print_array("bMatrix", size, bMatrix)
  call print_array("product", size, product)
end if

! print elapsed time
print 101, "Calculations took ", (time_end-time_start) / (count_rate/1000), "ms."

! cleanup
deallocate (aMatrix)
deallocate (bMatrix)
deallocate (product)

contains


! routine to initialize an array with data
subroutine init_array(row, col, off, arrSize, array)
  implicit none
  integer, intent(in) :: arrSize
  real, intent(inout) :: array(:,:)
  real, intent(in) :: row, col, off
  integer :: i,j;

  do i=1,arrSize
	do j=1,arrSize
		array(i,j) = row*i+col*j+off;
    end do
  end do
end subroutine init_array


! routine to print out contents of small arrays
subroutine print_array(name, arrSize, array)
  implicit none
  real, intent(inout) :: array(:,:)
  integer, intent(in) :: arrSize
  character*8, intent(in) :: name
  integer :: i,j;

  write (*,*) name
  do i=1,arrSize
	do j=1,arrSize
        write (*,"(f10.3,\)") array(i,j)
    end do
    write (*,*)
  end do

end subroutine print_array

! matrix multiply routine
!DEC$ ATTRIBUTES NOINLINE :: matrix_multiply
subroutine matrix_multiply(arrSize, aMatrix, bMatrix, product)
  implicit none
  real, intent(inout) :: product(:,:)
  real, intent(in)    :: aMatrix(:,:), bMatrix(:,:)
  integer, intent(in) :: arrSize
  integer :: i,j,k;
  real :: sum

  call annotate_site_begin("matrix_multiply")

  do i=1,arrSize
    call annotate_iteration_task("multiply_task")
    do j=1,arrSize
      sum=0
      do k=1,arrSize
        sum = sum + aMatrix(i,k) * bMatrix(k,j)
      end do
      product(i,j) = sum
    end do
  end do

  call annotate_site_end()

end subroutine matrix_multiply

end program MatrixMultiplication
