# SPDX-FileCopyrightText: Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import dbus
import dbus.service
import dbus.mainloop.glib
import logging
import threading
from enum import Enum

logger = logging.getLogger("main")

# Constants for D-Bus
DBUS_INTERFACE = "com.nvidia.nsight.GSTSignalInterface"
SIGNAL_NAME = "GSTSignalBroadcast"
OBJECT_PATH = "/com/nvidia/nsight/GSTSignalObject"
BUS_NAME = "com.nvidia.nsight.GSTSignalSender"

class SignalType(Enum):
    CONNECTION = "connection"
    RESIZE = "resize"
    SCALE = "scale"

class ConnectionStatus(Enum):
    CONNECTED = "connected"

class DbusManager:
    """
    Singleton class managing D-Bus communication.
    Handles sending and receiving signals via D-Bus.
    """
    _instance = None
    _lock = threading.Lock()

    def __new__(cls):
        with cls._lock:
            if cls._instance is None:
                cls._instance = super(DbusManager, cls).__new__(cls)
                cls._instance._initialized = False
        return cls._instance

    def __init__(self):
        if self._initialized:
            return
        self._initialized = True
        dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
        self.session_bus = dbus.SessionBus()
        self.bus_name = dbus.service.BusName(BUS_NAME, bus=self.session_bus)
        self.emitter = SignalEmitter(self.bus_name, OBJECT_PATH)

    def send_signal(self, message_dict):
        """
        Sends a generic signal with the given message dictionary.
        """
        if not isinstance(message_dict, dict):
            raise ValueError("message_dict must be a dictionary")
        self.emitter.GSTSignalBroadcast(message_dict)

    def send_connection_signal(self, peer_id):
        """
        Sends a signal indicating an initial connection.
        """
        self.send_signal({
            "type": SignalType.CONNECTION.value, 
            "status": ConnectionStatus.CONNECTED.value,
            "peer_id": peer_id
        })
        
    def send_resized_signal(self, width, height):
        """
        Sends a signal indicating a resize event with the given width and height.
        """
        self.send_signal({
            "type": SignalType.RESIZE.value, 
            "width": width, 
            "height": height
        })
        
    def send_scaled_signal(self, scale):
        """
        Sends a signal indicating a scale change with the given scale value.
        """
        self.send_signal({
            "type": SignalType.SCALE.value, 
            "scale": scale
        })

class SignalEmitter(dbus.service.Object):
    """
    D-Bus signal emitter class responsible for sending signals.
    """
    def __init__(self, bus_name, object_path):
        super().__init__(bus_name, object_path)

    @dbus.service.signal(dbus_interface=DBUS_INTERFACE, signature="a{sv}")
    def GSTSignalBroadcast(self, message_dict):
        """
        Emits a signal over D-Bus with the given message dictionary.
        """
        pass  # The signal is sent automatically by D-Bus