"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDecipherStream = void 0;
// @ts-ignore
const readable_stream_1 = require("readable-stream");
const material_management_node_1 = require("@aws-crypto/material-management-node");
const serialize_1 = require("@aws-crypto/serialize");
const verify_stream_1 = require("./verify_stream");
const fromUtf8 = (input) => Buffer.from(input, 'utf8');
const aadUtility = (0, serialize_1.aadFactory)(fromUtf8);
const PortableTransformWithType = readable_stream_1.Transform;
const ioTick = async () => new Promise((resolve) => setImmediate(resolve));
const noop = () => { }; // eslint-disable-line @typescript-eslint/no-empty-function
function getDecipherStream() {
    let decipherInfo;
    let decipherState = {};
    let pathologicalDrain = noop;
    let frameComplete = false;
    return new (class DecipherStream extends PortableTransformWithType {
        constructor() {
            super();
            this.on('pipe', (source) => {
                /* Precondition: The source must be a VerifyStream to emit the required events. */
                (0, material_management_node_1.needs)(source instanceof verify_stream_1.VerifyStream, 'Unsupported source');
                source
                    .once('DecipherInfo', (info) => {
                    decipherInfo = info;
                })
                    .on('BodyInfo', this._onBodyHeader)
                    .on('AuthTag', (authTag, next) => {
                    this._onAuthTag(authTag, next).catch((e) => this.emit('error', e));
                });
            });
        }
        _onBodyHeader = ({ iv, contentLength, sequenceNumber, isFinalFrame, }) => {
            /* Precondition: decipherInfo must be set before BodyInfo is sent. */
            (0, material_management_node_1.needs)(decipherInfo, 'Malformed State.');
            /* Precondition: Ciphertext must not be flowing before a BodyHeader is processed. */
            (0, material_management_node_1.needs)(!decipherState.decipher, 'Malformed State.');
            const { messageId, contentType, getDecipher } = decipherInfo;
            const aadString = aadUtility.messageAADContentString({
                contentType,
                isFinalFrame,
            });
            const messageAAD = aadUtility.messageAAD(messageId, aadString, sequenceNumber, contentLength);
            const decipher = getDecipher(iv).setAAD(Buffer.from(messageAAD.buffer, messageAAD.byteOffset, messageAAD.byteLength));
            const content = [];
            decipherState = { decipher, content, contentLength };
        };
        _transform(chunk, _encoding, callback) {
            /* Precondition: BodyHeader must be parsed before frame data. */
            (0, material_management_node_1.needs)(decipherState.decipher, 'Malformed State.');
            decipherState.contentLength -= chunk.length;
            /* Precondition: Only content should be transformed, so the lengths must always match.
             * The BodyHeader and AuthTag are striped in the VerifyStream and passed in
             * through events.  This means that if I receive a chunk without having reset
             * the content accumulation events are out of order.  Panic.
             */
            (0, material_management_node_1.needs)(decipherState.contentLength >= 0, 'Lengths do not match');
            const { content } = decipherState;
            content.push(chunk);
            if (decipherState.contentLength > 0) {
                // More data to this frame
                callback();
            }
            else {
                // The frame is full, waiting for `AuthTag`
                // event to decrypt and forward the clear frame
                frameComplete = callback;
            }
        }
        _read(size) {
            /* The _onAuthTag decrypts and pushes the encrypted frame.
             * If this.push returns false then this stream
             * should wait until the destination stream calls read.
             * This means that _onAuthTag needs to wait for some
             * indeterminate time.  I create a closure around
             * the resolution function for a promise that
             * is created in _onAuthTag.  This way
             * here in _read (the implementation of read)
             * if a frame is being pushed, we can release
             * it.
             */
            pathologicalDrain();
            pathologicalDrain = noop;
            super._read(size);
        }
        _onAuthTag = async (authTag, next) => {
            const { decipher, content, contentLength } = decipherState;
            /* Precondition: _onAuthTag must be called only after a frame has been accumulated.
             * However there is an edge case.  The final frame _can_ be zero length.
             * This means that _transform will never be called.
             */
            (0, material_management_node_1.needs)(frameComplete || contentLength === 0, 'AuthTag before frame.');
            /* Precondition UNTESTED: I must have received all content for this frame.
             * Both contentLength and frameComplete are private variables.
             * As such manipulating them separately outside of the _transform function
             * should not be possible.
             * I do not know of this condition would ever be false while the above is true.
             * But I do not want to remove the check as there may be a more complicated case
             * that makes this possible.
             * If such a case is found.
             * Write a test.
             */
            (0, material_management_node_1.needs)(contentLength === 0, 'Lengths do not match');
            // flush content from state.
            decipherState = {};
            decipher.setAuthTag(authTag);
            /* In Node.js versions 10.9 and older will fail to decrypt if decipher.update is not called.
             * https://github.com/nodejs/node/pull/22538 fixes this.
             */
            if (!content.length)
                decipher.update(Buffer.alloc(0));
            const clear = [];
            for (const cipherChunk of content) {
                const clearChunk = decipher.update(cipherChunk);
                clear.push(clearChunk);
                await ioTick();
            }
            // If the authTag is not valid this will throw
            const tail = decipher.final();
            clear.push(tail);
            for (const clearChunk of clear) {
                if (!this.push(clearChunk)) {
                    /* back pressure: if push returns false, wait until _read
                     * has been called.
                     */
                    await new Promise((resolve) => {
                        pathologicalDrain = resolve;
                    });
                }
            }
            /* This frame is complete.
             * Need to notify the VerifyStream continue.
             * See the note in `AuthTag` for details.
             * The short answer is that for small frame sizes,
             * the "next" frame associated auth tag may be
             * parsed and send before the "current" is processed.
             * This will cause the auth tag event to fire before
             * any _transform events fire and a 'Lengths do not match' precondition to fail.
             */
            next();
            // This frame is complete. Notify _transform to continue, see needs above for more details
            if (frameComplete)
                frameComplete();
            // reset for next frame.
            frameComplete = false;
        };
        _destroy() {
            // It is possible to have to destroy the stream before
            // decipherInfo is set.  Especially if the HeaderAuth
            // is not valid.
            decipherInfo && decipherInfo.dispose();
        }
    })();
}
exports.getDecipherStream = getDecipherStream;
//# sourceMappingURL=data:application/json;base64,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