/*
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "mb_ibm_at_io_decode.h"

struct cpssp {
	struct sig_cs *port_dma0_cs;
	struct sig_cs *port_dma1_cs;
	struct sig_cs *port_dmapage_cs;
	struct sig_cs *port_kbd_cs;
	struct sig_cs *port_ppi_cs;
	struct sig_cs *port_pit_cs;
	struct sig_cs *port_pic0_cs;
	struct sig_cs *port_pic1_cs;
	struct sig_cs *port_rtc_cs;
};

static int
mb_ibm_at_io_decode_inb(void *_cpssp, unsigned char *valp, uint32_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	/*
	 * IBM AT
	 * 1-28
	 */
	switch (port) {
	case 0x0000 ... 0x001f:
		/* DMA controller 1 */
		return sig_cs_readb(cpssp->port_dma0_cs, cpssp,
				port & 0x1f, valp);
	case 0x0020 ... 0x003f:
		/* Interrupt controller 1 */
		return sig_cs_readb(cpssp->port_pic0_cs, cpssp,
				port & 0x1f, valp);
	case 0x0040 ... 0x005f:
		/* Timer */
		return sig_cs_readb(cpssp->port_pit_cs, cpssp,
				port & 0x1f, valp);
	case 0x0060:
	case 0x0064:
		/* 8042 (Keyboard) */
		return sig_cs_readb(cpssp->port_kbd_cs, cpssp,
				port & 0xf, valp);
	case 0x0061:
	case 0x0063:
		/* PPI */
		return sig_cs_readb(cpssp->port_ppi_cs, cpssp,
				(port >> 1) & 1, valp);
	case 0x0070 ... 0x007f:
		/* Real-time clock, NMI */
		return sig_cs_readb(cpssp->port_rtc_cs, cpssp,
				port & 0xf, valp);
	case 0x0080 ... 0x009f:
		/* DMA page register */
		return sig_cs_readb(cpssp->port_dmapage_cs, cpssp,
				port, valp);
	case 0x00a0 ... 0x00bf:
		/* Interrupt controller 2 */
		return sig_cs_readb(cpssp->port_pic1_cs, cpssp,
				port & 0x1f, valp);
	case 0x00c0 ... 0x00df:
		/* DMA controller 2 */
		return sig_cs_readb(cpssp->port_dma1_cs, cpssp,
				(port >> 1) & 0x1f, valp);
	case 0x00f0 ... 0x00ff:
		/* Math Coprocessor */
		fixme();
		return 0;
	default:
		return -1;
	}
}

static int
mb_ibm_at_io_decode_outb(void *_cpssp, unsigned char val, uint32_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	/*
	 * IBM AT
	 * 1-28
	 */
	switch (port) {
	case 0x0000 ... 0x001f:
		/* DMA controller 1 */
		return sig_cs_writeb(cpssp->port_dma0_cs, cpssp,
				port & 0x1f, val);
	case 0x0020 ... 0x003f:
		/* Interrupt controller 1 */
		return sig_cs_writeb(cpssp->port_pic0_cs, cpssp,
				port & 0x1f, val);
	case 0x0040 ... 0x005f:
		/* Timer */
		return sig_cs_writeb(cpssp->port_pit_cs, cpssp,
				port & 0x1f, val);
	case 0x0060:
	case 0x0064:
		/* 8042 (Keyboard) */
		return sig_cs_writeb(cpssp->port_kbd_cs, cpssp,
				port & 0xf, val);
	case 0x0061:
	case 0x0063:
		/* PPI */
		return sig_cs_writeb(cpssp->port_ppi_cs, cpssp,
				(port >> 1) & 1, val);
	case 0x0070 ... 0x007f:
		/* Real-time clock, NMI */
		return sig_cs_writeb(cpssp->port_rtc_cs, cpssp,
				port & 0xf, val);
	case 0x0080 ... 0x009f:
		/* DMA page register */
		return sig_cs_writeb(cpssp->port_dmapage_cs, cpssp,
				port & 0x1f, val);
	case 0x00a0 ... 0x00bf:
		/* Interrupt controller 2 */
		return sig_cs_writeb(cpssp->port_pic1_cs, cpssp,
				port & 0x1f, val);
	case 0x00c0 ... 0x00df:
		/* DMA controller 2 */
		return sig_cs_writeb(cpssp->port_dma1_cs, cpssp,
				(port >> 1) & 0x1f, val);
	case 0x00f0 ... 0x00ff: /* Math Coprocessor */
		fixme();
		return 0;
	default:
		return -1;
	}
}

void *
mb_ibm_at_io_decode_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_isa_bus *port_bus,
	struct sig_cs *port_dma0_cs,
	struct sig_cs *port_dma1_cs,
	struct sig_cs *port_dmapage_cs,
	struct sig_cs *port_kbd_cs,
	struct sig_cs *port_ppi_cs,
	struct sig_cs *port_pit_cs,
	struct sig_cs *port_pic0_cs,
	struct sig_cs *port_pic1_cs,
	struct sig_cs *port_rtc_cs
)
{
	static const struct sig_isa_bus_funcs bus_funcs = {
		.inb = mb_ibm_at_io_decode_inb,
		.outb = mb_ibm_at_io_decode_outb,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->port_dma0_cs = port_dma0_cs;
	cpssp->port_dma1_cs = port_dma1_cs;
	cpssp->port_dmapage_cs = port_dmapage_cs;
	cpssp->port_kbd_cs = port_kbd_cs;
	cpssp->port_ppi_cs = port_ppi_cs;
	cpssp->port_pit_cs = port_pit_cs;
	cpssp->port_pic0_cs = port_pic0_cs;
	cpssp->port_pic1_cs = port_pic1_cs;
	cpssp->port_rtc_cs = port_rtc_cs;

	sig_isa_bus_connect(port_bus, cpssp, &bus_funcs);

	return cpssp;
}

void
mb_ibm_at_io_decode_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
mb_ibm_at_io_decode_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
mb_ibm_at_io_decode_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
