package pusher

import (
	"fmt"
	"os"
	"sync"
	"time"

	"pkg.deepin.io/daemon/sync/infrastructure/utils"

	"pkg.deepin.io/daemon/sync/infrastructure/log"

	"github.com/godbus/dbus"
	"github.com/godbus/dbus/introspect"
	"github.com/godbus/dbus/prop"
	nsq "github.com/nsqio/go-nsq"
)

const (
	dbusService = "com.deepin.pusher"
	dbusPath    = "/com/deepin/pusher"
	dbusIfc     = "com.deepin.pusher"
)

// Client connect to server to receive message
type Client struct {
	utils.DBusProxy

	notify    *notify
	running   bool
	mutex     sync.Mutex
	consumers []*nsq.Consumer

	lookupdHTTPAddrs string
	userInfo         *utils.UserInfo
}

func (c *notify) MakePropSpec() map[string]map[string]*prop.Prop {
	return map[string]map[string]*prop.Prop{
		c.GetDBusInterface(): {},
	}
}

// MakeSignals create signals
func (c *notify) MakeSignals() []introspect.Signal {
	return nil
}
func (c *notify) GetDBusName() string {
	return dbusService
}
func (c *notify) GetDBusPath() dbus.ObjectPath {
	return dbusPath
}
func (c *notify) GetDBusInterface() string {
	return dbusIfc
}

func nsqLookupdAddress() string {
	addr := os.Getenv("DEEPIN_PUSH_SERVER_API")
	if len(addr) == 0 {
		addr = "push.deepinid.deepin.com/hub/lookup"
	}
	return addr
}

// NewClient create a new nsq server
func NewClient() (*Client, error) {

	cli := &Client{
		notify:           newNotify(),
		lookupdHTTPAddrs: nsqLookupdAddress(),
		userInfo:         &utils.UserInfo{},
	}

	sessionBus, err := dbus.SessionBus()
	if nil != err {
		log.Error(err)
		return cli, err
	}
	return cli, cli.SetupDBus(sessionBus, cli.notify)
}

// Monitor user info change message
func (cli *Client) Monitor(conn *dbus.Conn) {
	// Receive login signal
	caller := conn.Object("com.deepin.deepinid", "/com/deepin/deepinid")
	caller.AddMatchSignal("org.freedesktop.DBus.Properties", "PropertiesChanged")
	c := make(chan *dbus.Signal, 100)
	conn.Signal(c)

	go func() {
		for v := range c {
			if len(v.Body) < 2 {
				continue
			}
			values := v.Body[1].(map[string]dbus.Variant)

			v, ok := values["UserInfo"]
			if !ok {
				continue
			}
			cli.userInfo.FromDBus(v.Value())

			log.Warningf("push state change to %v", cli.userInfo.IsValid())
			if cli.userInfo.IsValid() {
				cli.Stop()
				cli.Start()
			} else {
				cli.Stop()
			}
		}
	}()
}

// Start recvie message
func (cli *Client) Start() {
	log.Warningf("start pusher")
	go cli.startNSQ(fmt.Sprint(cli.userInfo.UserID), cli.userInfo.AccessToken, cli.userInfo.HardwareID)
}

// Stop recvie message
func (cli *Client) Stop() {
	log.Warningf("stop pusher")

	cli.mutex.Lock()
	cli.running = false
	for _, consumer := range cli.consumers {
		consumer.Stop()
	}
	for _, consumer := range cli.consumers {
		<-consumer.StopChan
	}
	cli.mutex.Unlock()
}

// startNSQ listen server info
func (cli *Client) startNSQ(userid, token, id string) {
	cli.mutex.Lock()
	if cli.running {
		log.Warningf("pusher has started!!!")
		cli.mutex.Unlock()
		return
	}
	cli.running = true
	cli.mutex.Unlock()

	userAgent := fmt.Sprintf("sync/%s go-nsq/%s", "1.0.0", nsq.VERSION)
	nsqdTCPAddrs := []string{}
	lookupdHTTPAddrs := []string{cli.lookupdHTTPAddrs}
	cli.listenMessage([]string{"user" + userid}, nsqdTCPAddrs, lookupdHTTPAddrs, id, userAgent, token)
}

func (cli *Client) listenMessage(topics, nsqdTCPAddrs, lookupdHTTPAddrs []string, channel, userAgent, token string) {
	cfg := nsq.NewConfig()

	cfg.UserAgent = userAgent
	cfg.AuthSecret = fmt.Sprintf("Deepin:%v", token)
	cfg.LookupdPollInterval = time.Minute * 5
	for i := 0; i < len(topics); i++ {
		log.Infof("Adding consumer for topic: %s %s\n", topics[i], channel)

		consumer, err := nsq.NewConsumer(topics[i], channel, cfg)
		if err != nil {
			log.Fatal(err)
		}

		consumer.AddHandler(&msgHandler{
			topicName: topics[i],
			notify:    cli.notify,
		})

		err = consumer.ConnectToNSQDs(nsqdTCPAddrs)
		if err != nil {
			log.Error(err)
		}

		err = consumer.ConnectToNSQLookupds(lookupdHTTPAddrs)
		if err != nil {
			log.Error(err)
		}

		cli.consumers = append(cli.consumers, consumer)
	}

}
