package network

import (
	"reflect"

	. "pkg.deepin.io/daemon/sync/modules/model"
)

type Connection struct {
	Type     string `json:"type"`
	Filename string `json:"filename"`
	Contents []byte `json:"contents"`
}
type ConnectionList []*Connection

type Data struct {
	Version     string         `json:"version"`
	Connections ConnectionList `json:"connections"`
}

const (
	Name = "network"
)

func (core *Data) Name() string {
	return Name
}

func (core *Data) Model() interface{} {
	return &Data{}
}

func (core *Data) ToIFC(v interface{}) DataIFC {
	return v.(*Data)
}

func (core *Data) Equal(target DataIFC) bool {
	info := target.(*Data)
	return core.Connections.Equal(info.Connections)
}

func (core *Data) GenCache(DataIFC) DataIFC {
	return core
}

func (core *Data) Merge(current, cache DataIFC) {
	curInfo := current.(*Data)
	cacheInfo := cache.(*Data)
	for _, conn := range curInfo.Connections {
		cacheTmp := cacheInfo.Connections.Get(conn.Type, conn.Filename)
		if cacheTmp != nil && reflect.DeepEqual(conn, cacheTmp) {
			// not changed
			continue
		}
		rTmp := core.Connections.Get(conn.Type, conn.Filename)
		if rTmp == nil {
			// added
			core.Connections = append(core.Connections, conn)
			continue
		}
		// changed
		rTmp.Contents = conn.Contents
	}
	// drop deleted connection
	for _, conn := range cacheInfo.Connections {
		curTmp := curInfo.Connections.Get(conn.Type, conn.Filename)
		if curTmp != nil {
			continue
		}
		// deleted
		core.Connections = core.Connections.Delete(conn.Type,
			conn.Filename)
	}
}

func (datas ConnectionList) Get(ty, filename string) *Connection {
	for _, data := range datas {
		if data.Type == ty && data.Filename == filename {
			return data
		}
	}
	return nil
}

func (datas ConnectionList) Delete(ty, filename string) ConnectionList {
	var ret ConnectionList
	for _, conn := range datas {
		if conn.Type == ty && conn.Filename == filename {
			continue
		}
		ret = append(ret, conn)
	}
	return ret
}

func (datas ConnectionList) Equal(list ConnectionList) bool {
	if len(datas) != len(list) {
		return false
	}
	for _, conn := range list {
		tmp := datas.Get(conn.Type, conn.Filename)
		if tmp == nil || !reflect.DeepEqual(tmp.Contents, conn.Contents) {
			println("[Network] not equal:", conn.Type, conn.Filename, tmp)
			return false
		}
	}
	return true
}
