/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"
#include "pstream_int.h"
#include "interpol.h"

static void
isosurface(double isoval, long nlev1, double *lev1, Field *field3D, Field *field2D)
{
  const size_t gridsize = gridInqSize(field3D->grid);
  const size_t nmiss = field3D->nmiss;
  const double missval = field3D->missval;
  const double *data3D = field3D->ptr;
  double *data2D = field2D->ptr;

  for (size_t i = 0; i < gridsize; ++i)
    {
      data2D[i] = missval;

      for (long k = 0; k < (nlev1 - 1); ++k)
        {
          const double val1 = data3D[k * gridsize + i];
          const double val2 = data3D[(k + 1) * gridsize + i];

          if (nmiss > 0)
            {
              const bool lmiss1 = DBL_IS_EQUAL(val1, missval);
              const bool lmiss2 = DBL_IS_EQUAL(val2, missval);
              if (lmiss1 && lmiss2) continue;
              if (lmiss1 && IS_EQUAL(isoval, val2)) data2D[i] = lev1[k + 1];
              if (lmiss2 && IS_EQUAL(isoval, val1)) data2D[i] = lev1[k];
              if (lmiss1 || lmiss2) continue;
            }

          if ((isoval >= val1 && isoval <= val2) || (isoval >= val2 && isoval <= val1))
            {
              data2D[i] = IS_EQUAL(val1, val2) ? lev1[k] : intlin(isoval, lev1[k], val1, lev1[k + 1], val2);
              break;
            }
        }
    }

  field2D->missval = missval;
  field2D->nmiss = arrayNumMV(gridsize, data2D, missval);
}

void *
Isosurface(void *process)
{
  int nrecs;
  int i;
  int varID, levelID;
  int zaxisID, zaxisID1 = -1;
  size_t nmiss;

  cdoInitialize(process);

  operatorInputArg("isoval");

  operatorCheckArgc(1);

  const double isoval = parameter2double(operatorArgv()[0]);

  if (Options::cdoVerbose) cdoPrint("Isoval: %g", isoval);

  const int streamID1 = cdoStreamOpenRead(cdoStreamName(0));

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const int nzaxis = vlistNzaxis(vlistID1);
  int nlevel = 0;
  for (i = 0; i < nzaxis; i++)
    {
      zaxisID = vlistZaxis(vlistID1, i);
      nlevel = zaxisInqSize(zaxisID);
      if (zaxisInqType(zaxisID) != ZAXIS_HYBRID && zaxisInqType(zaxisID) != ZAXIS_HYBRID_HALF)
        if (nlevel > 1)
          {
            zaxisID1 = zaxisID;
            break;
          }
    }
  if (i == nzaxis) cdoAbort("No processable variable found!");

  const int nlev1 = nlevel;
  std::vector<double> lev1(nlev1);
  cdoZaxisInqLevels(zaxisID1, &lev1[0]);

  const int zaxisIDsfc = zaxisCreate(ZAXIS_SURFACE, 1);
  for (i = 0; i < nzaxis; i++)
    if (zaxisID1 == vlistZaxis(vlistID1, i)) vlistChangeZaxisIndex(vlistID2, i, zaxisIDsfc);

  const int streamID2 = cdoStreamOpenWrite(cdoStreamName(1));
  pstreamDefVlist(streamID2, vlistID2);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  fieldInit(field);
  field.ptr = (double *) Malloc(gridsizemax * sizeof(double));

  const int nvars = vlistNvars(vlistID1);

  std::vector<bool> liso(nvars);
  std::vector<bool> vars(nvars);
  std::vector<Field> vars1(nvars);

  for (varID = 0; varID < nvars; varID++)
    {
      const int gridID = vlistInqVarGrid(vlistID1, varID);
      const int zaxisID = vlistInqVarZaxis(vlistID1, varID);
      const size_t gridsize = gridInqSize(gridID);
      const int nlevel = zaxisInqSize(zaxisID);
      const double missval = vlistInqVarMissval(vlistID1, varID);

      liso[varID] = (zaxisID == zaxisID1);

      fieldInit(vars1[varID]);
      vars1[varID].grid = gridID;
      vars1[varID].zaxis = zaxisID;
      vars1[varID].nmiss = 0;
      vars1[varID].missval = missval;
      vars1[varID].ptr = (double *) Malloc(gridsize * nlevel * sizeof(double));
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      pstreamDefTimestep(streamID2, tsID);

      for (varID = 0; varID < nvars; varID++)
        {
          vars[varID] = false;
          vars1[varID].nmiss = 0;
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          pstreamInqRecord(streamID1, &varID, &levelID);
          const size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
          const size_t offset = gridsize * levelID;
          double *single = vars1[varID].ptr + offset;

          pstreamReadRecord(streamID1, single, &nmiss);
          vars1[varID].nmiss += nmiss;
          vars[varID] = true;
        }

      for (varID = 0; varID < nvars; varID++)
        {
          if (vars[varID])
            {
              if (liso[varID])
                {
                  isosurface(isoval, nlev1, &lev1[0], &vars1[varID], &field);

                  pstreamDefRecord(streamID2, varID, 0);
                  pstreamWriteRecord(streamID2, field.ptr, field.nmiss);
                }
              else
                {
                  const size_t gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID));
                  const int nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));
                  const double missval = vlistInqVarMissval(vlistID2, varID);

                  for (levelID = 0; levelID < nlevel; levelID++)
                    {
                      const size_t offset = gridsize * levelID;
                      double *single = vars1[varID].ptr + offset;
                      nmiss = arrayNumMV(gridsize, single, missval);
                      pstreamDefRecord(streamID2, varID, levelID);
                      pstreamWriteRecord(streamID2, single, nmiss);
                    }
                }
            }
        }

      tsID++;
    }

  for (varID = 0; varID < nvars; varID++) Free(vars1[varID].ptr);

  Free(field.ptr);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
