#include "GhostUImg.h"
#include "utils/RsyncTask.h"
#include "utils/SquashfsTask.h"
#include "utils/CheckTask.h"
#include "utils/Utils.h"
#include "utils/Device.h"
#include "utils/FsTab.h"
#include <QDir>
#include <QDebug>
#include <QSettings>

const QString SystemUpgradeMountPoint = "/SystemUpgradeMountPoint";

GhostUImg::GhostUImg()
{
    m_subTimer = new QTimer(this);
    connect(m_subTimer, &QTimer::timeout, this, &GhostUImg::subTimeSlot);
}

GhostUImg::~GhostUImg()
{
    disconnect(m_subTimer, &QTimer::timeout, this, &GhostUImg::subTimeSlot);
}

ErrorCode GhostUImg::CheckGhostBackupDiskSpace(const QString &selectDir)
{
    m_destDir = selectDir;
    QJsonObject destPartInfoObj;
    QString err = "";
    if ( !Utils::getDestPartInfoByDir(m_destDir, destPartInfoObj, err)) {
        QJsonObject errJson;
        errJson.insert("errCode", UnKnow);
        errJson.insert("backupSizeBytes", 0);
        errJson.insert("liveFlag", 0);
        errJson.insert("errMsg", err);
        errJson.insert("uuid", "");
        errJson.insert("operateType", OperateType::CheckGhostBackupSpace);
        Q_EMIT spaceCheckFinished(Utils::JsonToQString(errJson));
        return UnKnow;
    }

    QString destUUID = destPartInfoObj.value("uuid").toString();

    if (nullptr == m_checkTask) {
        m_checkTask = new CheckTask(OperateType::CheckGhostBackupSpace, RecoveryType::Rsync);
        m_checkTask->setParent(this);
        connect(m_checkTask, &CheckTask::spaceCheckFinished, [=](const QJsonObject &jsonObject) {
            QJsonObject spaceJson = jsonObject;
            spaceJson.insert("operateType", OperateType::CheckGhostBackupSpace);
            spaceJson.insert("destDir", m_destDir);
            QString spaceInfo = Utils::JsonToQString(spaceJson);
            Q_EMIT spaceCheckFinished(spaceInfo);
        });

        connect(m_checkTask, &CheckTask::error, [=](const QJsonObject &jsonObject) {
            QJsonObject errJson = jsonObject;
            errJson.insert("errCode", UnKnow);
            errJson.insert("operateType", OperateType::CheckGhostBackupSpace);
            Q_EMIT spaceCheckFinished(Utils::JsonToQString(errJson));
        });
    }

    if (nullptr != m_checkTask) {
        m_checkTask->setDestPath(selectDir);
        m_checkTask->setDestUUID(destUUID);
        m_checkTask->start();
    }
    return OK;
}

ErrorCode GhostUImg::createUImg(const QString &backupDir, const qint64 totalSize)
{
    m_doSystemUpgradeConfig = Utils::isSystemUpgrade();

    // 另外一个根未挂载则挂载到SystemUpgradeMountPoint
    // TODO:如果系统盘下还有其他未挂载的分区，在制作ghsot镜像的时候也要将其挂载起来
    if (!doSystemUpgradeConfig()) {
        return UnKnow;
    }

    QJsonObject destPartInfoObj;
    QString err = "";
    if ( !Utils::getDestPartInfoByDir(backupDir, destPartInfoObj, err)) {
        QJsonObject errJson;
        errJson.insert("progress", 0);
        errJson.insert("remainSecond", 0);
        errJson.insert("errMsg", err);
        errJson.insert("operateType", OperateType::GhostBackup);
        Q_EMIT error(Utils::JsonToQString(errJson));
        return UnKnow;
    }

    // 计算总共需要的时间，按照2M/s的速度做时间预估
    m_totalTimeNeed = (totalSize * 1.0 / MiB) / 2;
    m_timeUsed = 0;

    QString backupDirMount = destPartInfoObj.value("mountpoint").toString();

    //每次备份都创建一个备份任务，任务执行完毕后自动删除
    m_rsyncTask = new RsyncTask;

    m_startTime = QDateTime::fromMSecsSinceEpoch(QDateTime::currentMSecsSinceEpoch()).toString("yyyyMMdd-hhmmss");
    m_destDir = backupDir + "/" + m_startTime + "/localhost";

    connect(m_rsyncTask, &RsyncTask::progressChanged, [=](const QJsonObject &jsonObject) {
        QJsonObject progress = jsonObject;
        progress.insert("operateType", OperateType::GhostBackup);
        int tmpProgress = progress["progress"].toInt() / 2;
        progress["remainSecond"] = updateTimeRemain(tmpProgress);
        progress["progress"] = tmpProgress;
        Q_EMIT progressChanged(Utils::JsonToQString(progress));
    });

    connect(m_rsyncTask, &RsyncTask::success, this,[this](const QJsonObject &jsonObject) {
        makeImgFile();
    }, Qt::QueuedConnection);

    connect(m_rsyncTask, &RsyncTask::error, [=](const QJsonObject &jsonObject) {
        // 失败场景下卸载之前挂载的SystemUpgradeMountPoint
        unDoSystemUpgradeConfig();
        QJsonObject errJson = jsonObject;
        errJson.insert("operateType", OperateType::GhostBackup);
        Q_EMIT error(Utils::JsonToQString(errJson));
    });

    connect(m_rsyncTask, &QThread::finished, [=] {
        m_rsyncTask->deleteLater();
    });

    m_rsyncTask->setOptions({"-aAXHi", "--verbose", "--delete", "--force", "--sparse", "--stats", "--no-inc-recursive", "--delete-excluded", "--info=progress2", "--ignore-missing-args"});
    m_rsyncTask->setSourcePath("/");
    m_rsyncTask->setDestinationPath(m_destDir);
    QStringList sysBackupExcludes = {"/cdrom/*", "/dev/*", "/proc/*", "/run/*", "/mnt/*", "/sys/*", "/media/*", "/tmp/*"};

#ifdef QT_DEBUG
    QStringList debugExclude = {
            "/var/*",
            "/opt/*",
            "/home/*",
            "/usr/*",
            "/persistent/var/*",
            "/persistent/opt/*",
            "/persistent/home/*",
            "/persistent/usr/*",
            "/data/var/*",
            "/data/opt/*",
            "/data/home/*",
            "/data/usr/*"
    };

 //   sysBackupExcludes << debugExclude;  // 方便快速调试
#endif

    QStringList rsyncSysBackupFilters = this->getRsyncSystemBackupFilters();
    if (!rsyncSysBackupFilters.isEmpty()) {
        sysBackupExcludes << rsyncSysBackupFilters;
    }

    QStringList ostreeSysBackupFilters = {"/osroot/*", "/data/osroot/*", "/persistent/osroot/*"};
    sysBackupExcludes << ostreeSysBackupFilters;

    QString varDir = "/var";
    FSTabInfoList fstabInfos = FSTab::getFSTabFromFile("/etc/fstab");
    QMap<QString, QString> bindDirMap;
    FSTab::getFstabBindDir(fstabInfos, bindDirMap);
    sysBackupExcludes << QString(varDir + "/lib/lxcfs/*");
    if (bindDirMap.keys().contains(varDir)) {
        sysBackupExcludes << QString(bindDirMap.value(varDir) + "lib/lxcfs/*");
    }

    sysBackupExcludes << QString(backupDir + "/" + m_startTime);
    sysBackupExcludes << QString(backupDir + "/" + m_startTime + "/*");
    sysBackupExcludes << QString(backupDirMount + backupDir + "/" + m_startTime);
    sysBackupExcludes << QString(backupDirMount + backupDir + "/" + m_startTime + "/*");

    // 如果是非升级场景，则过滤掉隐藏根中的数据
    if (!m_doSystemUpgradeConfig) {
        sysBackupExcludes << QString(SystemUpgradeMountPoint + "/*");
    }

    m_rsyncTask->setExcludes(sysBackupExcludes);
    m_rsyncTask->enableDryRun(false);

    QString linkDest = backupDir + "last/";
    m_rsyncTask->setLinkDest(linkDest);
    QDir dir(m_destDir);
    if (!dir.exists()) {
        dir.mkpath(m_destDir);
    }
    m_rsyncTask->buildArgumentsForBackup();
    m_rsyncTask->start();
    m_subTimer->start(1000);
    return OK;
}

void GhostUImg::clearUuidInRecoveryConf(const QString &recoveryConf)
{
    if (!QFile::exists(recoveryConf)) {
        qWarning()<<"clearUuidInRecoveryConf not exists recoveryConf = "<<recoveryConf;
        return;
    }

    QString emptyUuid = "";
    QSettings setConf(recoveryConf, QSettings::IniFormat);
    setConf.beginGroup(BACKUP_GROUP);
    setConf.setValue(BACKUP_DEVICE_UUID_KEY, emptyUuid);
    setConf.setValue(BACKUP_HISTORY_DEVICE_UUID_KEY, emptyUuid);
    setConf.endGroup();
}

void GhostUImg::makeImgFile()
{
    const QString srcDir = m_destDir.left(m_destDir.lastIndexOf("/"));
    QString ghostFileName = Utils::getGhostFileName(m_startTime);
    const QString imgFile = srcDir.left(srcDir.lastIndexOf("/") + 1) + ghostFileName;

    // ghost 安装后uuid会变，去掉配置文件里的uuid
    this->clearUuidInRecoveryConf(srcDir + "/localhost/etc/uos-recovery/uos-recovery.ini");

    // 保存设备信息
    if (!writeGhostDeviceInfo(srcDir)) {
        return;
    }

    backUpConfFiles();// 备份配置文件到.back
    unDoSystemUpgradeConfig();

    m_squashfsTask = new SquashfsTask;

    connect(m_squashfsTask, &SquashfsTask::progressChanged, [=](const QJsonObject &jsonObject) {
        QJsonObject progress = jsonObject;
        progress.insert("operateType", OperateType::GhostBackup);
        int tmpProgress = 50 + progress["progress"].toInt() / 2;
        progress["remainSecond"] = updateTimeRemain(tmpProgress);
        progress["progress"] = tmpProgress;
        Q_EMIT progressChanged(Utils::JsonToQString(progress));
    });

    connect(m_squashfsTask, &SquashfsTask::success, [=] {
        Process::spawnCmd("rm", {"-fr", srcDir});
        qInfo() << "SquashfsTask::success, delete srcDir = " << srcDir;
        // 普通用户可以清理ghost文件
        Process::spawnCmd("sudo", {"chmod", "o+rwx", imgFile});
        ResultInfo retInfo(0, "success", "");
        this->reportEventLog(retInfo, OperateType::GhostBackup, RecoveryType::Rsync);
        Q_EMIT success("");
    });

    connect(m_squashfsTask, &SquashfsTask::error, [=](const QJsonObject &jsonObject) {
        QJsonObject errJson = jsonObject;
        errJson.insert("operateType", OperateType::GhostBackup);
        Process::spawnCmd("rm", {"-fr", srcDir});
        Process::spawnCmd("rm", {"-fr", imgFile});
        qInfo() << "SquashfsTask::error, delete srcDir = " << srcDir;
        qInfo() << "SquashfsTask::error, delete imgFile = " << imgFile;

        int errCode = jsonObject.value("errCode").toInt(-1);
        QString errMsg = jsonObject.value("errMsg").toString();
        ResultInfo retInfo(errCode, "failed", errMsg);
        this->reportEventLog(retInfo, OperateType::GhostBackup, RecoveryType::Rsync);
        Q_EMIT error(Utils::JsonToQString(errJson));
    });

    connect(m_squashfsTask, &SquashfsTask::taskFinished, [=] {
        m_squashfsTask->deleteLater();
        m_subTimer->stop();
    });

    // 压缩文件系统
    if (!m_squashfsTask->makeSquashfs(srcDir, imgFile)) {
        return;
    }
}

int GhostUImg::updateTimeRemain(int progress)
{
    // 平滑处理时间进度，把总时间和当前进度 10 等分，根据实际进度值所在区间，更新已用时间值
    int tmpIndex = progress / 10;
    int perTimeItem = m_totalTimeNeed / 10;
    if (m_timeIndex != tmpIndex) {
        m_timeIndex = tmpIndex;
        m_timeUsed = perTimeItem * m_timeIndex;
    }

    if (m_timeUsed >= (perTimeItem * (m_timeIndex + 1))) {
        m_timeUsed = perTimeItem * (m_timeIndex + 1);
    }

    int remainSecond = m_totalTimeNeed - m_timeUsed;
    if (remainSecond < 0) {
        return 0;
    }

    return remainSecond;
}

bool GhostUImg::doSystemUpgradeConfig()
{
    QString rootDev = "";
    if (!getRootDev(rootDev)) {
        return false;
    }

    QJsonArray deviceInfos;
    QString err = "";
    if (!Utils::getLsblkJsonReturn(QString("lsblk %1 -lfpJ -o NAME,LABEL,MOUNTPOINT").arg(rootDev), deviceInfos, err)) {
        return false;
    }

    bool doneConfig = true;
    for (int i = 0; i < deviceInfos.size(); i++) {
        QJsonObject infoItem = deviceInfos.at(i).toObject();
        if (infoItem.value("label").toString().contains("Root")) {
            if (infoItem.value("mountpoint").toString().isEmpty()) {
                // 挂载当前的未挂载的根分区
                QDir dir(SystemUpgradeMountPoint);
                if (!dir.exists()) {
                    dir.mkpath(SystemUpgradeMountPoint);
                }

                QString cmd = QString("mount %1 %2").arg(infoItem.value("name").toString()).arg(SystemUpgradeMountPoint);
                QString cmdLog = "";
                err = "";
                doneConfig = Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err);
                break;
            }
        }
    }

    return doneConfig;
}

void GhostUImg::unDoSystemUpgradeConfig()
{
    QString cmd = QString("umount %1").arg(SystemUpgradeMountPoint);
    QString cmdLog = "";
    QString err = "";
    if (Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err)) {
        cmd = QString("rm -rf %1").arg(SystemUpgradeMountPoint);
        cmdLog = "";
        err = "";
        Process::spawnCmd("/bin/bash", {"-c", cmd}, cmdLog, err);
    }
}

bool GhostUImg::writeGhostDeviceInfo(const QString &filePath)
{
    // 保存设备信息到文件
    // 使用 uname -m 获取架构信息
    QString platform = "unKnow";
    QStringList infoList = Utils::getCmdListReturn("uname -m");
    if (infoList.size() > 0) {
        platform = infoList.first();
    }

    // 不能使用 lspci | grep VGA 获取显卡型号信息，安装器环境里没有lspci这个命令，改用lshw命令去获取
    QString displayDriver = "unKnow";
    QString vga;
    if (Utils::getVGAFromLshw(vga)) {
        displayDriver = vga;
        qInfo()<<"writeGhostDeviceInfo, displayDriver = "<<displayDriver;
    }

    // 不能使用 cat /proc/cpuinfo |grep "model" 获取cpu信号信息, 部分国产机器获取不到想要的数据
    QString cpuModel = "unKnow";
    CpuInfo cpu;
    if (Utils::getCpuInfo(cpu)) {
        cpuModel = cpu.modelName;
        qInfo()<<"writeGhostDeviceInfo, cpuModel = "<<cpuModel;
    }

    QString fstabPath = "/etc/fstab";
    // 根据fstab中的设备id，找到对应的磁盘设备读取磁盘大小信息
    QList<GhostDiskInfo> diskInfoList;
    Device::getGhostDiskInfo(fstabPath, diskInfoList);
    quint64 diskTotalSize = 0;
    QJsonArray sysDiskJsonArray;
    for (auto &diskInfo : diskInfoList) {
        diskTotalSize += diskInfo.totalSizeBytes;
        QJsonObject jsonObject = diskInfo.marshal();
        sysDiskJsonArray.append(jsonObject);
    }
    // qint64 usedDeviceSize = getUsedDeviceSize(fstabPath);

    // 根据fstab获取bind信息
    QJsonArray bindInfosArray = getFstabInfos(fstabPath);

    // 判断引导模式
    bool isEFI = Utils::isEFIMode();

    // 保存以上信息到srcDir/info.json文件中
    QJsonObject infosObj;
    infosObj.insert("platform", platform);
    infosObj.insert("displayDriver", displayDriver);
    infosObj.insert("cpuModel", cpuModel);
    infosObj.insert("diskSize", QString("%1").arg(diskTotalSize));
    infosObj.insert("diskInfo", sysDiskJsonArray);
    infosObj.insert("fstab", bindInfosArray);
    infosObj.insert("isEFI", isEFI);
    infosObj.insert("systemUpgradeInfo", getSystemUpgradeInfo());

    QJsonDocument infosDoc(infosObj);
    QFile infosFile(filePath + "/info.json");
    if (!infosFile.open(QFile::WriteOnly)) {
        return false;
    }

    infosFile.write(infosDoc.toJson());
    infosFile.close();

    // 按照partition_policy.json格式生成分区配置文件
    QString err = "";
    return Device::exportPartitionInfoByLsblk(filePath + "/partition_policy.json", {}, err);
}

qint64 GhostUImg::getUsedDeviceSize(const QString &fstabPath)
{
    // 根据fstab中的设备id，找到对应的磁盘设备读取磁盘大小信息
    QStringList usedPartsUuids = {};
    FSTabInfoList fstabInfos = FSTab::getFSTabFromFile(fstabPath);
    for (FSTabInfoPtr fstabInfoItem : fstabInfos) {
        QString fstabDeviceItem = fstabInfoItem->device;
        if (!fstabDeviceItem.isEmpty()) {
            fstabDeviceItem = fstabDeviceItem.right(fstabDeviceItem.length() - fstabDeviceItem.indexOf("=") - 1);
            usedPartsUuids.append(fstabDeviceItem);
        }
    }

    QStringList usedDevicesName = {};
    QStringList usedLVMPartsName = {};
    Device usedDev;
    DeviceInfoList usedDevices = usedDev.getDeviceByLsblk();
    for (DeviceInfoPtr deviceItem : usedDevices) {
        if (deviceItem->uuid.isEmpty()) {
            continue;
        }
        if (usedPartsUuids.indexOf(deviceItem->uuid) != -1) {
            if (!deviceItem->type.compare("part")) {
                if (usedDevicesName.indexOf(deviceItem->pkname) == -1) {
                    usedDevicesName.append(deviceItem->pkname);
                }
            }
            if (!deviceItem->type.compare("lvm")) {
                if (usedLVMPartsName.indexOf(deviceItem->pkname) == -1) {
                    usedLVMPartsName.append(deviceItem->pkname);
                }
            }
        }
    }

    for (DeviceInfoPtr deviceItem : usedDevices) {
        if (usedLVMPartsName.contains(deviceItem->name)) {
            if (usedDevicesName.indexOf(deviceItem->pkname) == -1) {
                usedDevicesName.append(deviceItem->pkname);
            }
        }
    }

    qint64 usedDeviceSize = 0;
    for (QString usedDeviceItem : usedDevicesName) {
        usedDev.getDeviceByName(usedDeviceItem);
        usedDeviceSize += usedDev.getDeviceInfo()->sizeBytes;
    }

    return usedDeviceSize;
}

QJsonArray GhostUImg::getFstabInfos(const QString &fstabPath)
{
    QJsonArray bindInfosArray;
    FSTabInfoList fstabInfos = FSTab::getFSTabFromFile(fstabPath);
    for (FSTabInfoPtr fstabInfoItem : fstabInfos) {
        if (!fstabInfoItem->device.compare("")) {
            continue;
        }

        QJsonObject infosObj;
        infosObj.insert("device", fstabInfoItem->device);
        infosObj.insert("mountPoint", fstabInfoItem->mountPoint);
        infosObj.insert("type", fstabInfoItem->type);
        infosObj.insert("options", fstabInfoItem->options);
        infosObj.insert("dump", fstabInfoItem->dump);
        infosObj.insert("pass", fstabInfoItem->pass);
        bindInfosArray.append(infosObj);
    }

    return bindInfosArray;
}

bool GhostUImg::getRootDev(QString &rootDev)
{
    rootDev = "";

    QJsonArray fstabInfosArray = getFstabInfos("/etc/fstab");
    if (fstabInfosArray.size() < 0) {
        return false;
    }

    QString rootUuid = "";
    for (int i = 0; i < fstabInfosArray.size(); i++) {
        QJsonObject fstabInfoItem = fstabInfosArray.at(i).toObject();
        if (!fstabInfoItem.value("mountPoint").toString().compare("/")) {
            rootUuid = fstabInfoItem.value("device").toString();
            rootUuid = rootUuid.right(rootUuid.length() - rootUuid.indexOf("=") - 1);
            break;
        }
    }

    if (rootUuid.isEmpty()) {
        return false;
    }

    QJsonArray deviceInfos;
    QString err = "";
    if (!Utils::getLsblkJsonReturn("lsblk -lfpJ -o PKNAME,UUID", deviceInfos, err)) {
        return false;
    }

    for (int i = 0; i < deviceInfos.size(); i++) {
        QJsonObject infoItem = deviceInfos.at(i).toObject();
        if (!infoItem.value("uuid").toString().compare(rootUuid)) {
            rootDev = infoItem.value("pkname").toString();
            break;
        }
    }

    if (rootDev.isEmpty()) {
        return false;
    }

    return  true;
}


QJsonObject GhostUImg::getSystemUpgradeInfo()
{
    QJsonObject systemUpgradeInfo;
    systemUpgradeInfo.insert("isSystemUpgrade", m_doSystemUpgradeConfig);

    QJsonArray partInfo;
    QString err = "";
    if (!Utils::getLsblkJsonReturn("lsblk -lfpJ -o NAME,UUID,MOUNTPOINT", partInfo, err)) {
        return systemUpgradeInfo;
    }

    for (int i = 0; i < partInfo.size(); ++i) {
        QJsonObject infoObject = partInfo.at(i).toObject();
        QString mountPoint = infoObject.value("mountpoint").toString();
        if (!mountPoint.compare(SystemUpgradeMountPoint)) {
            systemUpgradeInfo.insert("rootPath", infoObject.value("name").toString());
            systemUpgradeInfo.insert("rootUuid", infoObject.value("uuid").toString());
            systemUpgradeInfo.insert("mountPoint", SystemUpgradeMountPoint);
            break;
        }
    }

    return systemUpgradeInfo;
}

void GhostUImg::backUpConfFiles()
{
    QJsonArray deviceInfos;
    QString err = "";
    if (!Utils::getLsblkJsonReturn("lsblk -lfpJ -o UUID,MOUNTPOINT", deviceInfos, err)) {
        return;
    }

    QString ruleFilePath = "/etc/udev/rules.d/80-udisks-installer.rules";
    // 备份m_destDir/etc/udev/rules.d/80-udisks-installer.rules文件到.back
    replaceRootUuidContent(m_destDir + ruleFilePath,
                           m_destDir + ruleFilePath + ".back",
                           deviceInfos);

    QString abRecoveryConfFilePath = "/etc/deepin/ab-recovery.json";
    // 备份m_destDir/etc/deepin/ab-recovery.json文件到.back
    replaceRootUuidContent(m_destDir + abRecoveryConfFilePath,
                             m_destDir + abRecoveryConfFilePath + ".back",
                             deviceInfos);

    if (m_doSystemUpgradeConfig) {
        // 备份m_destDir/SystemUpgradeMountPoint/etc/udev/rules.d/80-udisks-installer.rules到.back
        replaceRootUuidContent(m_destDir + SystemUpgradeMountPoint + ruleFilePath,
                               m_destDir + SystemUpgradeMountPoint + ruleFilePath + ".back",
                               deviceInfos);

        // 备份m_destDir/SystemUpgradeMountPoint/etc/deepin/ab-recovery.json到.back
        replaceRootUuidContent(m_destDir + SystemUpgradeMountPoint + abRecoveryConfFilePath,
                                 m_destDir + SystemUpgradeMountPoint + abRecoveryConfFilePath + ".back",
                                 deviceInfos);
    }
}

void GhostUImg::replaceRootUuidContent(const QString &srcFilePath, const QString &destFilePath, const QJsonArray &devInfos)
{
    QFile ruleFile(srcFilePath);
    if (!ruleFile.open(QIODevice::Text | QIODevice::ReadOnly)) {
        return;
    }
    QString data = ruleFile.readAll();
    ruleFile.close();

    for (int i = 0; i < devInfos.size(); i++) {
        QJsonObject infoItem = devInfos.at(i).toObject();
        QString partUuid = infoItem.value("uuid").toString();
        QString partMountPoint = infoItem.value("mountpoint").toString();
        if (!partMountPoint.compare("/")) {
            data.replace(partUuid, "/target");
        } else {
            data.replace(partUuid, partMountPoint);
        }
    }

    QFile ruleFileBack(destFilePath);
    if (!ruleFileBack.open(QIODevice::Text | QIODevice::WriteOnly)) {
        return;
    }
    ruleFileBack.write(data.toLocal8Bit());
    ruleFileBack.close();

    QFile::remove(srcFilePath);
}

void GhostUImg::subTimeSlot()
{
    ++m_timeUsed;
}

QStringList GhostUImg::getRsyncSystemBackupFilters()
{
    QStringList totalExcludes;

    const QString uosRecoveryConf = "/" + UOS_RECOVERY_INI;
    QStringList sysBackupUuidList = Utils::getAllSystemBackupUUID(uosRecoveryConf);
    if (sysBackupUuidList.isEmpty()) {
        qWarning()<<"getRsyncSystemBackupFilters sysBackupUuidList isEmpty";
        return totalExcludes;
    }

    const QString sysBackupFilterPrefix = "/backup/system/snapshots/*";
    for (auto &uuid : sysBackupUuidList) {
        QString mountPoint;
        if(!Device::getMountPointByUUID(uuid, mountPoint)) {
            continue;
        }

        if (mountPoint == "/") {
            totalExcludes << sysBackupFilterPrefix;
        } else {
            totalExcludes << (mountPoint + sysBackupFilterPrefix);
        }
    }

    return totalExcludes;
}


#ifdef QT_DEBUG
QStringList GhostUImg::getRsyncSystemBackupFiltersTest()
{
    return this->getRsyncSystemBackupFilters();
}
#endif
