// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "runtimestate.h"

#include <QStandardPaths>
#include <QFileInfo>
#include <QJsonDocument>
#include <QJsonObject>
#include <QVariant>
#include <QDir>

#include <unistd.h>

GLOBAL_USE_NAMESPACE

RuntimeState::RuntimeState(const QString &m) : model(m)
{

}

int RuntimeState::pid() const
{
    auto runFile = stateFile();
    if (!QFileInfo::exists(runFile))
        return -1;

    QFile file(runFile);
    if (!file.open(QFile::ReadOnly))
        return -1;

    auto doc = QJsonDocument::fromJson(file.readAll());
    file.close();

    auto vh = doc.object().toVariantHash();
    if (vh.contains(kRuntime_state_pid)) {
        bool ok = false;
        int pid = vh.value(kRuntime_state_pid).toInt(&ok);
        if (ok && QFileInfo::exists(QString("/proc/%0").arg(pid))) {
            return pid;
        }
    }

    return -1;
}

QString RuntimeState::stateFile() const
{
    QString runFile = QString("%0/%1.state").arg(stateDir()).arg(model);
    return runFile;
}

QString RuntimeState::stateDir()
{
    QString rundir = QStandardPaths::standardLocations(QStandardPaths::TempLocation).first();
    rundir = rundir + QString("/%0-%1").arg(EXE_NAME).arg(getuid());
    return rundir;
}

void RuntimeState::mkpath()
{
    auto dir = stateDir();
    if (!QFileInfo::exists(dir))
        QDir().mkpath(dir + "/.");
}

void RuntimeState::writeSate(QFile *file, const QVariantHash &state)
{
    if (!file)
        return;

    // clear content
    file->seek(0);
    file->resize(0);

    auto obj = QJsonObject::fromVariantHash(state);
    file->write(QJsonDocument(obj).toJson());
    file->flush();
}

QList<QVariantHash> RuntimeState::listAll()
{
    QList<QVariantHash> ret;
    QDir dir(stateDir());
    auto files = dir.entryList({"*.state"});

    for (auto path : files) {
        QFile file(dir.absoluteFilePath(path));
        if (!file.open(QFile::ReadOnly))
            continue;
        auto doc = QJsonDocument::fromJson(file.readAll());
        auto vh = doc.object().toVariantHash();
        if (vh.contains(kRuntime_state_pid)) {
            bool ok = false;
            int pid = vh.value(kRuntime_state_pid).toInt(&ok);
            if (ok && QFileInfo::exists(QString("/proc/%0").arg(pid))) {
                ret.append(vh);
            }
        }
    }

    return ret;
}
