/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "glue.h"

#include "sig_floppy.h"

int
sig_floppy_wp(
	struct sig_floppy *b,
	void *s
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *);

		if (nr == b->member_count) {
			return 1; /* Write protected */
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->wp;
		if (func) {
			return func(b->member[nr].s);
		}
	}
}

int
sig_floppy_read(
	struct sig_floppy *b,
	void *s,
	unsigned int blk,
	uint8_t *data,
	uint8_t *trackp,
	uint8_t *sectorp
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int,
				uint8_t *, uint8_t *, uint8_t *);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->read;
		if (func) {
			return func(b->member[nr].s, blk, data, trackp, sectorp);
		}
	}
}

int
sig_floppy_write(
	struct sig_floppy *b,
	void *s,
	unsigned int blk,
	const uint8_t *data,
	const uint8_t track,
	const uint8_t sector
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		int (*func)(void *, unsigned int,
				const uint8_t *, uint8_t, uint8_t);

		if (nr == b->member_count) {
			return -1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->write;
		if (func) {
			return func(b->member[nr].s, blk, data, track, sector);
		}
	}
}

void
sig_floppy_connect(
	struct sig_floppy *b,
	void *s,
	const struct sig_floppy_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_FLOPPY);
	assert(b->member_count < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->member_count].s = s;
	b->member[b->member_count].f = f;
	b->member_count++;
}

void
sig_floppy_disconnect(
	struct sig_floppy *b,
	void *s
)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_FLOPPY);

	for (nr = 0; ; nr++) {
		assert(nr < sizeof(b->member) / sizeof(b->member[0]));
		if (b->member[nr].s == s) {
			break;
		}
	}
	b->member_count--;
	b->member[nr].s = b->member[b->member_count].s;
	b->member[nr].f = b->member[b->member_count].f;
}

static int
sig_floppy_s0_wp(
	void *_f
)
{
	struct sig_floppy_merge *f = _f;

	return sig_floppy_wp(f->s1, f);
}

static int
sig_floppy_s0_read(
	void *_f,
	unsigned int blk,
	uint8_t *data,
	uint8_t *trackp,
	uint8_t *sectorp
)
{
	struct sig_floppy_merge *f = _f;

	return sig_floppy_read(f->s1, f, blk, data, trackp, sectorp);
}

static int
sig_floppy_s0_write(
	void *_f,
	unsigned int blk,
	const uint8_t *data,
	uint8_t track,
	uint8_t sector
)
{
	struct sig_floppy_merge *f = _f;

	return sig_floppy_write(f->s1, f, blk, data, track, sector);
}

static int
sig_floppy_s1_wp(
	void *_f
)
{
	struct sig_floppy_merge *f = _f;

	return sig_floppy_wp(f->s0, f);
}

static int
sig_floppy_s1_read(
	void *_f,
	unsigned int blk,
	uint8_t *data,
	uint8_t *trackp,
	uint8_t *sectorp
)
{
	struct sig_floppy_merge *f = _f;

	return sig_floppy_read(f->s0, f, blk, data, trackp, sectorp);
}

static int
sig_floppy_s1_write(
	void *_f,
	unsigned int blk,
	const uint8_t *data,
	uint8_t track,
	uint8_t sector
)
{
	struct sig_floppy_merge *f = _f;

	return sig_floppy_write(f->s0, f, blk, data, track, sector);
}

struct sig_floppy_merge *
sig_floppy_merge(
	struct sig_floppy *s0,
	struct sig_floppy *s1
)
{
	static const struct sig_floppy_funcs s0_funcs = {
		.wp = sig_floppy_s0_wp,
		.read = sig_floppy_s0_read,
		.write = sig_floppy_s0_write,
	};
	static const struct sig_floppy_funcs s1_funcs = {
		.wp = sig_floppy_s1_wp,
		.read = sig_floppy_s1_read,
		.write = sig_floppy_s1_write,
	};
	struct sig_floppy_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	/* Out */
	m->s0 = s0;
	m->s1 = s1;

	/* In */
	sig_floppy_connect(s0, m, &s0_funcs);
	sig_floppy_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_floppy_split(struct sig_floppy_merge *m)
{
	sig_floppy_disconnect(m->s0, m);
	sig_floppy_disconnect(m->s1, m);
}

struct sig_floppy *
sig_floppy_create(const char *name)
{
	struct sig_floppy *sig;

	sig = shm_alloc(sizeof(*sig));
	assert(sig);

	sig->type = SIG_GEN_FLOPPY;
	sig->member_count = 0;

	return sig;
}

void
sig_floppy_destroy(struct sig_floppy *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_FLOPPY);

	shm_free(sig);
}

void
sig_floppy_suspend(struct sig_floppy *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_floppy_resume(struct sig_floppy *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}
