/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE

#include <string.h>
#include <errno.h>

#endif /* INCLUDE */
#ifdef STATE

struct {
	FILE *file;
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static bool
NAME_(open_file)(struct cpssp *cpssp, const char *path_to_file);
/*forward*/ static void
NAME_(read_commands_from_file)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(close_file)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static bool
NAME_(open_file)(struct cpssp *cpssp, const char *path_to_file)
{
	if (path_to_file == NULL) {
		return false;
	}

	cpssp->NAME.file = fopen(path_to_file, "r");
	if (cpssp->NAME.file == NULL) {
		perror(path_to_file);
		return false;
	}
	return true;

}

static bool
parse_value_int(char *value_string, int *value)
{
	char *end;
	long int val = strtol(value_string, &end, 10);
	if (value_string != end) {
		*value = (int) val;
		return true;
	}
	return false;
}

static bool
parse_value_double(char *value_string, double *value)
{
	char *end;
	double val = strtod(value_string, &end);
	if (value_string != end) {
		*value = val;
		return true;
	}
	return false;
}

static bool
parse_value_trigger(char *value_string, int *channel, bool *rising_edge)
{
	int char_pos;
	int num_pos;
	int edge_pos;
	int end_pos;
	int value_length;
	char c;
	bool analog;

	value_length = strlen(value_string);
	char channel_number[value_length];
	char edge[value_length];
	char *end;
	memset(channel_number, '\0', value_length);
	memset(edge, '\0', value_length);
	num_pos = -1;
	edge_pos = -1;
	end_pos = -1;
	for (char_pos = 0; char_pos < value_length; char_pos++) {
		c = value_string[char_pos];
		if (c == '_') {
			edge_pos = char_pos + 1;
		} else if ('0' <= c && c <= '9'
			 && num_pos == -1) {
			num_pos = char_pos;
		} else if (c == '\0' || c == '\n' || c == '\t') {
			end_pos = char_pos;
			break;
		}
	}
	if (num_pos == -1
		|| edge_pos == -1
		|| end_pos == -1) {
		return false;
	}
	if (value_string[0] == 'C') {
		analog = true;
	} else {
		analog = false;
	}
	strncpy(channel_number, value_string + num_pos, edge_pos - num_pos - 1);
	strncpy(edge, value_string + edge_pos, end_pos - edge_pos);

	if (strcmp(edge, "PE") == 0) {
		*rising_edge = true;
	} else if (strcmp(edge, "NE") == 0) {
		*rising_edge = false;
	}
	long int chan = strtol(channel_number, &end, 10);
	if (analog == false) {
		chan += NUMBER_ANALOG_CHANNELS;
	}
	if (value_string != end && 0 <= chan) {
		*channel = (int) chan;
		return true;
	}

	return false;
}

static char *
NAME_(get_next_subcommand)(
	struct cpssp *cpssp,
	char *command,
	int start_pos,
	int *end_pos,
	int *channel
)
{
	int length;
	int char_pos;
	int num_pos;
	int sub_w_pos;
	char c;

	char_pos = start_pos;
	num_pos = -1;
	sub_w_pos = 0;
	length = strlen(command);
	char sub_command[length];
	memset(sub_command, '\0', length);
	while (char_pos < length) {
		c = command[char_pos];
		if ('A' <= c
			&& c <= 'Z') {
			sub_command[sub_w_pos++] = c;
			if (num_pos != -1) {
				/* channel numbers always at the end */
				return NULL;
			}
		} else if ( '0' <= c
			&& c <= '9'
			&& num_pos == -1) {
			/* mark beginning of digits */
			num_pos = char_pos;
		} else if (c == ':') {
			if (char_pos != start_pos) {
				break;
			} else {
				sub_command[sub_w_pos++] = c;
			}
		} else if (c == '\0') {
			break;
		}
		char_pos++;
	}
	if (start_pos == length) {
		return NULL;
	}
	if (num_pos != -1) {
		/* set channel */
		int w_pos;
		int r_pos;
		char *end_ptr;
		char digit_part[char_pos - num_pos];
		r_pos = num_pos;
		w_pos = 0;
		while (r_pos < char_pos) {
			c = command[r_pos];
			if (c < '0' || c > '9') {
				fprintf(stderr, "Not a valid scpi command\n");
				return NULL;
			}
			digit_part[w_pos++] = command[r_pos];
			r_pos++;
		}
		*channel =(int)strtol(digit_part, &end_ptr, 10);
		sub_command[num_pos] = '\0';
	} else {
		sub_command[char_pos] = '\0';
	}
	*end_pos = char_pos;
	return strdup(sub_command);
}

static void
NAME_(parse_command)(struct cpssp *cpssp, char *command)
{
	int channel;
	int start_pos;
	int end_pos;
	char *command_tokens[2];
	int command_length;
	char *p;
	int token;

	start_pos = 0;

	/* TODO new strtok layer, separating by semicolon */
	command_length = strlen(command);
	p = strtok(command, " \t");
	token = 0;
	channel = -1;
	while (p != NULL) {
		command_tokens[token++] = strdup(p);
		p = strtok(NULL, " \t");
	}

	char *sub_command = NAME_(get_next_subcommand)(cpssp, command_tokens[0],
			start_pos, &end_pos, &channel);

	char reduced_command[command_length];
	memset(reduced_command, '\0', command_length);
	while (sub_command != NULL) {
		/*
		 * short form is usually 4 characters.
		 * those which are more than 4 characters long
		 * and the last character is a vowel only 3
		 */
		int short_form_length;
		if (sub_command[0] == ':') {
			short_form_length = 5;
		} else {
			short_form_length = 4;
		}
		if (strlen(sub_command) > short_form_length
		 && (sub_command[short_form_length - 1] == 'A'
		  || sub_command[short_form_length - 1] == 'E'
		  || sub_command[short_form_length - 1] == 'I'
		  || sub_command[short_form_length - 1] == 'U'
		  || sub_command[short_form_length - 1] == 'O')) {
			sub_command[short_form_length - 1] = '\0';
		} else {
			sub_command[short_form_length] = '\0';
		}
		strcat(reduced_command, sub_command);
		/*fprintf(stderr, "sub %s\n", sub_command);*/
		start_pos = end_pos;
		sub_command = NAME_(get_next_subcommand)(cpssp, command_tokens[0],
				start_pos, &end_pos, &channel);
	}
	if (reduced_command[0] != ':') {
		/* TODO command relative to last command tree */
	}

	if (strcmp(reduced_command, ":DISP:TRAC:Y:PDIV") == 0) {
		/* mV/div */
		int value;
		char channel_type;
		if (token == 2
			&& parse_value_int(command_tokens[1], &value) == true) {

			if (channel == -1) {
				channel = 1;
			}
			channel_type = get_channel_type(cpssp, channel);
			if (channel_type == 'v') {
				volt_div_set_sim(cpssp, value, channel);
			} else if(channel_type == 'c') {
				amp_div_set_sim(cpssp, value, channel);
			}
		}
	} else if (strcmp(reduced_command, ":DISP:TRAC:X:PDIV") == 0) {
		/* mS/div */
		unsigned int value; /* value in ns */
		if (token == 2
		 && parse_value_int(command_tokens[1], &value) == true) {
			sec_div_set_sim(cpssp, value);
		}
	} else if (strcmp(reduced_command, ":SWE:TIME") == 0) {
		/* mS/div */
		int value;
		if (token == 2
		 && parse_value_int(command_tokens[1], &value) == true) {
			if (channel == -1) {
				channel = 0;
			}
			cpssp->sweep_time = (double)value / 1000.0;
		}
	} else if (strcmp(reduced_command, ":ACQ:TRIG") == 0) {
		int channel = -1;
		bool rising_edge = false;
		fprintf(stderr, "trigger!!!\n");
		if (token == 2
		 && parse_value_trigger(command_tokens[1],
				&channel, &rising_edge) == true) {
			trigger_channel_set_sim(cpssp, channel);
			trigger_edge_set_sim(cpssp, rising_edge);
			/*trigger_set_sim(cpssp, channel, rising_edge);*/
		}
	} else if (strcmp(reduced_command, ":CHAN:OFFS") == 0) {
		double value;
		if (token == 2
		 && parse_value_double(command_tokens[1], &value) == true) {
			if (channel == -1) {
				channel = 0;
			}
			offset_set_sim(cpssp, channel, value);

		}

	} else if (strcmp(reduced_command, ":DIG:OFFS") == 0) {
		double value;
		if (token == 2
		 && parse_value_double(command_tokens[1], &value) == true) {
			if (channel == -1) {
				channel = 0;
			}
			offset_set_sim(cpssp, channel + NUMBER_ANALOG_CHANNELS, value);
		}
	} else if (strcmp(reduced_command, ":ACQ:SRAT") == 0) {
		int value;
		if (token == 2
		 && parse_value_int(command_tokens[1], &value) == true) {
			sample_rate_set_sim(cpssp, value);
		}

	}
}

static void
NAME_(read_commands_from_file)(struct cpssp *cpssp)
{
	char line[1024];

	while (fgets(line, sizeof(line), cpssp->NAME.file) != NULL) {
		NAME_(parse_command)(cpssp, line);
	}
}

static void
NAME_(close_file)(struct cpssp *cpssp)
{
	if (cpssp->NAME.file != NULL) {
		fclose(cpssp->NAME.file);
	}
}

#endif /* BEHAVIOUR */
