/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <libepubgen/libepubgen.h>

#include <libwps/libwps.h>

#include "EpubPackage.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "wps2epub"

using namespace libwps;

class EpubOutputFileHelper
{
public:
	EpubOutputFileHelper(const char *const outFileName, const char *password, const char *encoding) :
		m_fileName(outFileName), m_password(password), m_encoding(encoding)
	{
	}

	bool convertDocument(librevenge::RVNGInputStream &input)
	{
		EpubPackage package(m_fileName);
		libepubgen::EPUBTextGenerator collector(&package);

		libwps::WPSResult result=WPSDocument::parse(&input, &collector, m_password, m_encoding);
		if (result == WPS_ENCRYPTION_ERROR)
		{
			fprintf(stderr, "ERROR : encrypted file, problem with password\n");
			return false;
		}
		else if (result != WPS_OK)
		{
			fprintf(stderr, "ERROR : Couldn't convert the document\n");
			return false;
		}

		return true;
	}

	bool isSupportedFormat(librevenge::RVNGInputStream &input)
	{
		WPSCreator creator;
		WPSKind kind = WPS_TEXT;
		bool needEncoding;
		WPSConfidence confidence = WPSDocument::isFileFormatSupported(&input, kind, creator, needEncoding);
		if (confidence == WPS_CONFIDENCE_NONE || kind != WPS_TEXT)
			return false;
		if (!m_encoding && needEncoding)
			fprintf(stderr, "WARNING: called without encoding, try to decode with basic character set encoding.\n");

		return true;
	}

private:
	std::string m_fileName;
	//! the password
	char const *m_password;
	//! the encoding
	char const *m_encoding;
};

int printUsage(const char *name)
{
	UsageHelper usage(name, "converts MS Works documents to ePub.", "[OPTIONS] INPUT OUTPUT");
	usage.addEncodingOption();
	usage.addListEncodingsOption();
	usage.addPasswordOption();
	return usage.printUsage();
}

int main(int argc, char *argv[])
{
	char *szInputFile = 0;
	char *szOutFile = 0;
	char *encoding = 0;
	char *password = 0;

	for (int i = 1; i < argc; i++)
	{
		if (strcmp(argv[i], "--encoding")==0)
		{
			if (i+1>=argc)
				return printUsage(TOOLNAME);
			encoding=argv[++i];
			continue;
		}
		else if (strcmp(argv[i], "--password")==0)
		{
			if (i+1>=argc)
				return printUsage(TOOLNAME);
			password=argv[++i];
			continue;
		}
		else if (strcmp(argv[i], "--list-encodings")==0)
			return UsageHelper::printEncodings();
		else if (strcmp(argv[i], "--version")==0)
			return UsageHelper::printVersion(TOOLNAME);
		else if (strcmp(argv[i], "--help")==0)
			return printUsage(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile || !szOutFile)
		return printUsage(TOOLNAME);

	librevenge::RVNGFileStream input(szInputFile);
	EpubOutputFileHelper helper(szOutFile, password, encoding);

	if (!helper.isSupportedFormat(input))
	{
		fprintf(stderr, "ERROR: the document is not in any supported format.\n");
		return 1;
	}

	if (!helper.convertDocument(input))
		return 1;

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
