/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libepubgen project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "EPUBTextElements.h"

namespace libepubgen
{

class EPUBTextElement
{
public:
  EPUBTextElement() {}
  virtual ~EPUBTextElement() {}
  virtual void write(librevenge::RVNGTextInterface *iface) const = 0;
};

namespace
{

class CloseCommentElement : public EPUBTextElement
{
public:
  CloseCommentElement() {}
  ~CloseCommentElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
};

class CloseEndnoteElement : public EPUBTextElement
{
public:
  CloseEndnoteElement() {}
  ~CloseEndnoteElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
};

class CloseFooterElement : public EPUBTextElement
{
public:
  CloseFooterElement() {}
  ~CloseFooterElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
};

class CloseFootnoteElement : public EPUBTextElement
{
public:
  CloseFootnoteElement() {}
  ~CloseFootnoteElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
};

class CloseFrameElement : public EPUBTextElement
{
public:
  CloseFrameElement() {}
  ~CloseFrameElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
};

class CloseHeaderElement : public EPUBTextElement
{
public:
  CloseHeaderElement() {}
  ~CloseHeaderElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
};

class CloseLinkElement : public EPUBTextElement
{
public:
  CloseLinkElement() {}
  ~CloseLinkElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseLinkElement();
  }
};

class CloseListElementElement : public EPUBTextElement
{
public:
  CloseListElementElement() {}
  ~CloseListElementElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseListElementElement();
  }
};

class CloseOrderedListLevelElement : public EPUBTextElement
{
public:
  CloseOrderedListLevelElement() {}
  ~CloseOrderedListLevelElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseOrderedListLevelElement();
  }
};

class CloseParagraphElement : public EPUBTextElement
{
public:
  CloseParagraphElement() {}
  ~CloseParagraphElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseParagraphElement();
  }
};

class CloseSpanElement : public EPUBTextElement
{
public:
  CloseSpanElement() {}
  ~CloseSpanElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseSpanElement();
  }
};

class CloseTableElement : public EPUBTextElement
{
public:
  CloseTableElement() {}
  ~CloseTableElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseTableElement();
  }
};

class CloseTableCellElement : public EPUBTextElement
{
public:
  CloseTableCellElement() {}
  ~CloseTableCellElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseTableCellElement();
  }
};

class CloseTableRowElement : public EPUBTextElement
{
public:
  CloseTableRowElement() {}
  ~CloseTableRowElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseTableRowElement();
  }
};

class CloseUnorderedListLevelElement : public EPUBTextElement
{
public:
  CloseUnorderedListLevelElement() {}
  ~CloseUnorderedListLevelElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new CloseUnorderedListLevelElement();
  }
};

class InsertBinaryObjectElement : public EPUBTextElement
{
public:
  InsertBinaryObjectElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~InsertBinaryObjectElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertBinaryObjectElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class InsertCoveredTableCellElement : public EPUBTextElement
{
public:
  InsertCoveredTableCellElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~InsertCoveredTableCellElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertCoveredTableCellElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class InsertEquationElement : public EPUBTextElement
{
public:
  InsertEquationElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~InsertEquationElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertEquationElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class InsertFieldElement : public EPUBTextElement
{
public:
  InsertFieldElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~InsertFieldElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertFieldElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class InsertLineBreakElement : public EPUBTextElement
{
public:
  InsertLineBreakElement() {}
  ~InsertLineBreakElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertLineBreakElement();
  }
};

class InsertSpaceElement : public EPUBTextElement
{
public:
  InsertSpaceElement() {}
  ~InsertSpaceElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertSpaceElement();
  }
};

class InsertTabElement : public EPUBTextElement
{
public:
  InsertTabElement() {}
  ~InsertTabElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertTabElement();
  }
};

class InsertTextElement : public EPUBTextElement
{
public:
  InsertTextElement(const librevenge::RVNGString &text) :
    m_text(text) {}
  ~InsertTextElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new InsertTextElement(m_text);
  }
private:
  librevenge::RVNGString m_text;
};

class OpenCommentElement : public EPUBTextElement
{
public:
  OpenCommentElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenCommentElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new OpenCommentElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenEndnoteElement : public EPUBTextElement
{
public:
  OpenEndnoteElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenEndnoteElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new OpenEndnoteElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenFooterElement : public EPUBTextElement
{
public:
  OpenFooterElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenFooterElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new OpenFooterElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenFootnoteElement : public EPUBTextElement
{
public:
  OpenFootnoteElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenFootnoteElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new OpenFootnoteElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenFrameElement : public EPUBTextElement
{
public:
  OpenFrameElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenFrameElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new OpenFrameElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenHeaderElement : public EPUBTextElement
{
public:
  OpenHeaderElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenHeaderElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
  EPUBTextElement *clone()
  {
    return new OpenHeaderElement(m_propList);
  }
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenLinkElement : public EPUBTextElement
{
public:
  OpenLinkElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenLinkElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenListElementElement : public EPUBTextElement
{
public:
  OpenListElementElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenListElementElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenOrderedListLevelElement : public EPUBTextElement
{
public:
  OpenOrderedListLevelElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenOrderedListLevelElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenParagraphElement : public EPUBTextElement
{
public:
  OpenParagraphElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenParagraphElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenSpanElement : public EPUBTextElement
{
public:
  OpenSpanElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenSpanElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenTableElement : public EPUBTextElement
{
public:
  OpenTableElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenTableElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenTableCellElement : public EPUBTextElement
{
public:
  OpenTableCellElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenTableCellElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenTableRowElement : public EPUBTextElement
{
public:
  OpenTableRowElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenTableRowElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

class OpenUnorderedListLevelElement : public EPUBTextElement
{
public:
  OpenUnorderedListLevelElement(const librevenge::RVNGPropertyList &propList) :
    m_propList(propList) {}
  ~OpenUnorderedListLevelElement() {}
  void write(librevenge::RVNGTextInterface *iface) const;
private:
  librevenge::RVNGPropertyList m_propList;
};

void CloseCommentElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeComment();
}

void CloseEndnoteElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeEndnote();
}

void CloseFooterElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeFooter();
}

void CloseFootnoteElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeFootnote();
}

void CloseFrameElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeFrame();
}


void CloseHeaderElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeHeader();
}

void CloseLinkElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeLink();
}

void CloseListElementElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeListElement();
}

void CloseOrderedListLevelElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeOrderedListLevel();
}

void CloseParagraphElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeParagraph();
}

void CloseSpanElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeSpan();
}

void CloseTableElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeTable();
}

void CloseTableCellElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeTableCell();
}

void CloseTableRowElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeTableRow();
}

void CloseUnorderedListLevelElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->closeUnorderedListLevel();
}

void InsertBinaryObjectElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertBinaryObject(m_propList);
}

void InsertCoveredTableCellElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertCoveredTableCell(m_propList);
}

void InsertEquationElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertEquation(m_propList);
}

void InsertFieldElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertField(m_propList);
}

void InsertLineBreakElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertLineBreak();
}

void InsertSpaceElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertSpace();
}

void InsertTabElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertTab();
}

void InsertTextElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->insertText(m_text);
}

void OpenCommentElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openComment(m_propList);
}

void OpenEndnoteElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openEndnote(m_propList);
}

void OpenFooterElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openFooter(m_propList);
}

void OpenFootnoteElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openFootnote(m_propList);
}

void OpenFrameElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openFrame(m_propList);
}

void OpenHeaderElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openHeader(m_propList);
}

void OpenLinkElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openLink(m_propList);
}

void OpenListElementElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openListElement(m_propList);
}

void OpenOrderedListLevelElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openOrderedListLevel(m_propList);
}

void OpenParagraphElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openParagraph(m_propList);
}

void OpenSpanElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openSpan(m_propList);
}

void OpenTableElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openTable(m_propList);
}

void OpenTableCellElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openTableCell(m_propList);
}

void OpenTableRowElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openTableRow(m_propList);
}

void OpenUnorderedListLevelElement::write(librevenge::RVNGTextInterface *iface) const
{
  if (iface)
    iface->openUnorderedListLevel(m_propList);
}

}

EPUBTextElements::EPUBTextElements()
  : m_elements()
{
}

EPUBTextElements::~EPUBTextElements()
{
  for (std::deque<EPUBTextElement *>::const_iterator iter = m_elements.begin(); iter != m_elements.end(); ++iter)
    delete *iter;
}

void EPUBTextElements::write(librevenge::RVNGTextInterface *iface) const
{
  for (std::deque<EPUBTextElement *>::const_iterator iter = m_elements.begin(); iter != m_elements.end(); ++iter)
    (*iter)->write(iface);
}

void EPUBTextElements::addCloseComment()
{
  m_elements.push_back(new CloseCommentElement());
}

void EPUBTextElements::addCloseEndnote()
{
  m_elements.push_back(new CloseEndnoteElement());
}

void EPUBTextElements::addCloseFooter()
{
  m_elements.push_back(new CloseFooterElement());
}

void EPUBTextElements::addCloseFootnote()
{
  m_elements.push_back(new CloseFootnoteElement());
}

void EPUBTextElements::addCloseFrame()
{
  m_elements.push_back(new CloseFrameElement());
}

void EPUBTextElements::addCloseHeader()
{
  m_elements.push_back(new CloseHeaderElement());
}

void EPUBTextElements::addCloseLink()
{
  m_elements.push_back(new CloseLinkElement());
}

void EPUBTextElements::addCloseListElement()
{
  m_elements.push_back(new CloseListElementElement());
}

void EPUBTextElements::addCloseOrderedListLevel()
{
  m_elements.push_back(new CloseOrderedListLevelElement());
}

void EPUBTextElements::addCloseParagraph()
{
  m_elements.push_back(new CloseParagraphElement());
}

void EPUBTextElements::addCloseSpan()
{
  m_elements.push_back(new CloseSpanElement());
}

void EPUBTextElements::addCloseTable()
{
  m_elements.push_back(new CloseTableElement());
}

void EPUBTextElements::addCloseTableCell()
{
  m_elements.push_back(new CloseTableCellElement());
}

void EPUBTextElements::addCloseTableRow()
{
  m_elements.push_back(new CloseTableRowElement());
}

void EPUBTextElements::addCloseUnorderedListLevel()
{
  m_elements.push_back(new CloseUnorderedListLevelElement());
}

void EPUBTextElements::addInsertBinaryObject(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new InsertBinaryObjectElement(propList));
}

void EPUBTextElements::addInsertCoveredTableCell(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new InsertCoveredTableCellElement(propList));
}

void EPUBTextElements::addInsertEquation(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new InsertEquationElement(propList));
}

void EPUBTextElements::addInsertField(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new InsertFieldElement(propList));
}

void EPUBTextElements::addInsertLineBreak()
{
  m_elements.push_back(new InsertLineBreakElement());
}

void EPUBTextElements::addInsertSpace()
{
  m_elements.push_back(new InsertSpaceElement());
}

void EPUBTextElements::addInsertTab()
{
  m_elements.push_back(new InsertTabElement());
}

void EPUBTextElements::addInsertText(const librevenge::RVNGString &text)
{
  m_elements.push_back(new InsertTextElement(text));
}

void EPUBTextElements::addOpenComment(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenCommentElement(propList));
}

void EPUBTextElements::addOpenEndnote(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenEndnoteElement(propList));
}

void EPUBTextElements::addOpenFooter(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenFooterElement(propList));
}

void EPUBTextElements::addOpenFootnote(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenFootnoteElement(propList));
}

void EPUBTextElements::addOpenHeader(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenHeaderElement(propList));
}

void EPUBTextElements::addOpenFrame(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenFrameElement(propList));
}

void EPUBTextElements::addOpenLink(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenLinkElement(propList));
}

void EPUBTextElements::addOpenListElement(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenListElementElement(propList));
}

void EPUBTextElements::addOpenOrderedListLevel(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenOrderedListLevelElement(propList));
}

void EPUBTextElements::addOpenParagraph(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenParagraphElement(propList));
}

void EPUBTextElements::addOpenSpan(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenSpanElement(propList));
}

void EPUBTextElements::addOpenTable(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenTableElement(propList));
}

void EPUBTextElements::addOpenTableCell(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenTableCellElement(propList));
}

void EPUBTextElements::addOpenTableRow(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenTableRowElement(propList));
}

void EPUBTextElements::addOpenUnorderedListLevel(const librevenge::RVNGPropertyList &propList)
{
  m_elements.push_back(new OpenUnorderedListLevelElement(propList));
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
