(************************************************************************)
(*   FlexDLL                                                            *)
(*   Alain Frisch                                                       *)
(*                                                                      *)
(*   Copyright 2007 Institut National de Recherche en Informatique et   *)
(*   en Automatique.                                                    *)
(************************************************************************)

(* The main application: parse COFF files,
   compute relocation and export tables, rewrite some COFF files,
   call the native linker *)

open Coff
open Cmdline

let search_path = ref []
let default_libs = ref []

let gcc = ref "gcc"
let objdump = ref "objdump"

let is_crt_lib = function
  | "LIBCMT"
  | "MSVCRT" -> true
  | _ -> false

let flexdir =
  try
    let s = Sys.getenv "FLEXDIR" in
    if s = "" then raise Not_found else s
  with Not_found ->
    Filename.dirname Sys.executable_name

let ext_obj () =
  if !toolchain = `MSVC || !toolchain = `MSVC64 then ".obj" else ".o"

(* Temporary files *)

let temps = ref []

let add_temp fn =
  temps := fn :: !temps; fn

let temp_file s x =
  add_temp (Filename.temp_file s x)

let open_temp_file s x =
  let (f, c) = Filename.open_temp_file s x in (add_temp f, c)

let safe_remove s =
  try Sys.remove s
  with Sys_error _ -> ()

let () =
  at_exit
    (fun () -> if not !save_temps then List.iter safe_remove !temps)

(* Calling external commands *)

let read_file fn =
  let ic = open_in fn in
  let r = ref [] in
  (try while true do r := input_line ic :: !r done with End_of_file -> ());
  close_in ic;
  List.rev !r


let get_output ?(use_bash = false) ?(accept_error=false) cmd =
  let fn = Filename.temp_file "flexdll" "" in
  let cmd' = cmd ^ " > " ^ (Filename.quote fn) in
    if String.length cmd' < 8182 && not use_bash then
      begin
        if (Sys.command cmd' <> 0) && not accept_error
        then failwith ("Cannot run " ^ cmd);
      end
    else
      begin
        let (cfn, oc) = open_temp_file "longcmd" ".sh" in
          output_string oc cmd'; close_out oc;
          if Sys.command (Printf.sprintf "bash %s" cfn) <> 0
          then failwith ("Cannot run " ^ cmd)
      end;
    let r = read_file fn in
      Sys.remove fn;
      r

let get_output1 ?use_bash cmd =
  match get_output ?use_bash cmd with
  | output::_ -> output
  | [] -> raise (Failure ("command " ^ cmd ^ " did not return any output"))


(* Preparing command line *)

let mk_dirs_opt pr = String.concat " " (List.map (fun s -> pr ^ (Filename.quote s)) !dirs)

exception Not_utf8

let utf8_next s i =
  let fail () = raise Not_utf8 in
  let check i =
    if i >= String.length s then fail ();
    let n = Char.code s.[i] in
    if n lsr 6 <> 0b10 then fail () else n
  in
  if !i >= String.length s then fail ();
  match s.[!i] with
  | '\000'..'\127' as c ->
      let n = Char.code c in
      i := !i + 1;
      n
  | '\192'..'\223' as c ->
      let n1 = Char.code c in
      let n2 = check (!i+1) in
      let n =
        ((n1 land 0b11111) lsl 6) lor
        ((n2 land 0b111111))
      in
      i := !i + 2;
      n
  | '\224'..'\239' as c ->
      let n1 = Char.code c in
      let n2 = check (!i+1) in
      let n3 = check (!i+2) in
      let n =
        ((n1 land 0b1111) lsl 12) lor
        ((n2 land 0b111111) lsl 6) lor
        ((n3 land 0b111111))
      in
      i := !i + 3;
      n
  | '\240'..'\247' as c ->
      let n1 = Char.code c in
      let n2 = check (!i+1) in
      let n3 = check (!i+2) in
      let n4 = check (!i+3) in
      let n =
        ((n1 land 0b111) lsl 18) lor
        ((n2 land 0b111111) lsl 12) lor
        ((n3 land 0b111111) lsl 6) lor
        ((n4 land 0b111111))
      in
      i := !i + 4;
      n
  | _ ->
      fail ()

let toutf16 s =
  let i = ref 0 in
  let b = Buffer.create (String.length s * 2) in
  while !i < String.length s do
    Buffer.add_utf_16le_uchar b (Uchar.unsafe_of_int (utf8_next s i))
  done;
  Buffer.contents b

(* Build @responsefile to work around Windows limitations on
   command-line length *)
let build_diversion lst =
  let responsefile = temp_file "camlresp" "" in
  let oc = open_out_bin responsefile in
  let lst =
    List.map (fun f ->
        let s = Bytes.of_string (Filename.quote f) in
        for i = 0 to Bytes.length s - 1 do
          if Bytes.get s i = '\\' then Bytes.set s i '/'
        done;
        Bytes.to_string s ^ "\r\n"
      ) (List.filter (fun f -> f <> "") lst)
  in
  let utf16, lst =
    match List.map toutf16 lst with
    | lst ->
        true, lst
    | exception Not_utf8 ->
        false, lst
  in
  if utf16 then output_string oc "\xFF\xFE"; (* LE BOM *)
  List.iter (fun s -> output_string oc s) lst;
  close_out oc;
  "@" ^ responsefile

type cmdline = {
    may_use_response_file: bool;
  }

let new_cmdline () =
  let rf = match !toolchain with
  | `MSVC | `MSVC64 | `LIGHTLD -> true
  | `MINGW | `MINGW64 | `GNAT | `CYGWIN | `CYGWIN64 -> false
  in
  {
   may_use_response_file = rf;
  }

let run_command cmdline cmd =
  let cmd_quiet =
    match !toolchain with
    | `MSVC | `MSVC64 when !verbose < 1 -> cmd ^ " >NUL"
    | _ -> cmd
  in
  (* note: for Cygwin, using bash allow to follow symlinks to find
     gcc... *)
  if !toolchain = `CYGWIN || !toolchain = `CYGWIN64 ||
     String.length cmd_quiet >= 8192
  then begin
    (* Dump the command in a text file and apply bash to it. *)
    let (fn, oc) = open_temp_file "longcmd" "" in
    output_string oc cmd;
    close_out oc;

    if !verbose >= 1 then Printf.printf "(call with bash: %s)\n%!" fn;
    if Sys.command (Printf.sprintf "bash %s" fn) <> 0 then
      failwith "Error during linking\n"
  end else
    if Sys.command cmd_quiet <> 0 then begin
      if cmd <> cmd_quiet then ignore (Sys.command cmd);
      failwith "Error during linking\n"
    end

let quote_files cmdline lst =
  let s =
    String.concat " "
      (List.map (fun f -> if f = "" then f else Filename.quote f) lst) in
  if String.length s >= 1024 && cmdline.may_use_response_file then Filename.quote (build_diversion lst)
  else s


(* Looking for files *)

let cygpath l =
  get_output (Printf.sprintf "cygpath -m %s" (String.concat " " (List.map Filename.quote l)))

let cygpath1 fn =
  get_output1 (Printf.sprintf "cygpath -m %s" fn)

let file_exists fn =
  if Sys.file_exists fn && not (Sys.is_directory fn) then Some fn
  else if !use_cygpath && Sys.file_exists (fn ^ ".lnk") then
    Some (cygpath1 fn)
  else None

let rec find_file_in = function
  | [] -> None
  | fn::rest ->
      match file_exists fn with
        | Some x -> Some x
        | None -> find_file_in rest

let find_file fn =
  let l =
    List.flatten
      (List.map
         (fun dir ->
            let fn = Filename.concat dir fn in
            [ fn; fn ^ ".lib"; fn ^ ".dll.a"; fn ^ ".a" ]
         ) (""::!search_path)) in
  match find_file_in l with
    | Some x -> Some x
    | None -> if !use_cygpath then find_file_in (cygpath l) else None

let rec map_until_found f = function
  | [] ->
      None
  | x::xs ->
      match f x with
      | None ->
          map_until_found f xs
      | r ->
          r

let find_file =
  let memo = Hashtbl.create 16 in
  fun fn ->
    let k = String.lowercase_ascii fn in
    try Hashtbl.find memo k
    with Not_found ->
      try Hashtbl.find memo (k ^ ".lib")
      with Not_found ->
        let fns =
          if String.length fn > 2 && String.sub fn 0 2 = "-l" then
            let base = String.sub fn 2 (String.length fn - 2) in
            if !toolchain = `MSVC || !toolchain = `MSVC64 then
              ["lib" ^ base; base]
            else
              ["lib" ^ base]
          else [fn] in
        let r =
          match map_until_found find_file fns with
          | Some fn -> fn
          | None ->
              failwith (Printf.sprintf "Cannot find file %S" fn)
        in
        Hashtbl.add memo k r;
        Hashtbl.add memo (k ^ ".lib") r;
        r


(*******************************)

let int32_to_buf b i =
  Buffer.add_char b (Char.chr (i land 0xff));
  Buffer.add_char b (Char.chr ((i lsr 8) land 0xff));
  Buffer.add_char b (Char.chr ((i lsr 16) land 0xff));
  Buffer.add_char b (Char.chr ((i lsr 24) land 0xff))

let int_to_buf b i =
  assert(i >= 0);
  match !machine with
  | `x86 -> int32_to_buf b i
  | `x64 -> int32_to_buf b i; int32_to_buf b 0

let exportable s =
  match !machine with
  | `x86 ->
      s <> "" && (s.[0] = '_'  || s.[0] = '?')
  | `x64 ->
      if String.length s > 2 && s.[0] = '?' && s.[1] = '?' then false
      else true

let drop_underscore obj s =
  match !machine with
  | `x86 ->
      assert (s <> "");
      begin
        match s.[0] with
        | '_' -> String.sub s 1 (String.length s - 1)
        | '?' -> s
        | _ -> failwith (Printf.sprintf "In %s, symbol %s doesn't start with _ or ?" obj.obj_name s)
      end
  | `x64 ->
      s

let has_prefix pr s =
  String.length s > String.length pr && String.sub s 0 (String.length pr) = pr

let check_prefix pr s =
  if has_prefix pr s then
    Some (String.sub s (String.length pr) (String.length s - String.length pr))
  else None

let parse_libpath s =
  let n = String.length s in
  let rec aux l =
    if l >= n then []
    else
      try
        let i = String.index_from s l ';' in
        String.sub s l (i - l) :: aux (succ i)
      with Not_found -> [ String.sub s l (n - l) ]
  in
  aux 0

module StrSet = Set.Make(String)

(* Put all the relocations on the symbols defined by a predicate
   into a relocation table. A relocation table describes how to
   patch some addresses with the value of some external symbols (given
   by their name). It also lists segments that are normally write-protected
   and that must be de-protected to enable the patching process. *)

let add_reloc_table obj obj_name p =
  let sname = Symbol.gen_sym () in (* symbol pointing to the reloc table *)
  let sect = Section.create ".reltbl" 0xc0300040l in
  let data = Buffer.create 1024 in
  let strings = Buffer.create 1024 in
  let nonwr = ref [] in
  let nonwrsym = Symbol.intern sect 0l in
  let strsym = Symbol.intern sect 0l in
  let str_pos = Hashtbl.create 16 in

  Reloc.abs !machine sect 0l nonwrsym;
  int_to_buf data 0;

  (* TODO: use a single symbol per section *)
  let syms = ref [] in
  let reloc secsym min max rel =
    if p rel.symbol then (
      (* kind *)
      let kind = match !machine, rel.rtype with
        | `x86, 0x06 (* IMAGE_REL_I386_DIR32 *)
        | `x64, 0x01 (* IMAGE_REL_AMD64_ADDR64 *) ->
            0x0002 (* absolute, native size (32/64) *)

        | `x64, 0x04 (* IMAGE_REL_AMD64_REL32 *)
        | `x86, 0x14 (* IMAGE_REL_I386_REL32 *) when not !no_rel_relocs ->
            0x0001 (* rel32 *)

        | `x64, 0x05 when not !no_rel_relocs -> 0x0004 (* rel32_1 *)
        | `x64, 0x08 when not !no_rel_relocs-> 0x0003 (* rel32_4 *)
        | `x64, 0x06 when not !no_rel_relocs-> 0x0005 (* rel32_2 *)

        | `x86, (0x0a (* IMAGE_REL_I386_SECTION *) |
                 0x0b (* IMAGE_REL_I386_SECREL*) ) ->
            0x0100 (* debug relocs: ignore *)

        | _, k  ->
            let msg =
              Printf.sprintf "Unsupported relocation kind %04x for %s in %s"
                k rel.symbol.sym_name obj_name
            in
            failwith msg
(*            Printf.eprintf "%s\n" msg;
            0x0001 *)
      in
      int_to_buf data kind;

      (* name *)
      let name = drop_underscore obj rel.symbol.sym_name in
      let pos =
        try Hashtbl.find str_pos name
        with Not_found ->
          let pos = Buffer.length strings in
          Hashtbl.add str_pos name pos;
          Buffer.add_string strings name;
          Buffer.add_char strings '\000';
          pos
      in
      Reloc.abs !machine sect (Int32.of_int (Buffer.length data)) strsym;
      int_to_buf data pos;

      Reloc.abs !machine sect (Int32.of_int (Buffer.length data))
        (Lazy.force secsym);
      int_to_buf data (Int32.to_int rel.addr);

      if rel.addr <= !min then min := rel.addr;
      if rel.addr >= !max then max := rel.addr;
      false
    ) else true
  in
  let section sec =
    if sec.sec_opts &&& 0x1000l <> 0l && has_prefix ".rdata$.refptr." sec.sec_name then
      begin
        (* under Cygwin64, gcc introduces mergable (link once) COMDAT sections to store
           indirection pointers to external darta symbols.  Since we don't deal with such section
           properly, we turn them into regular data section, thus loosing sharing (but we don't care). *)
        sec.sec_opts <- 0xc0500040l;
        sec.sec_name <- Printf.sprintf ".flexrefptrsection%i" (Oo.id (object end));
      end;

    let min = ref Int32.max_int and max = ref Int32.min_int in
    let sym = lazy (let s = Symbol.intern sec 0l in
                    syms := s :: !syms;
                    s) in

    sec.relocs <- filter (reloc sym min max) sec.relocs;
    if (sec.sec_opts &&& 0x80000000l = 0l) && !min <= !max then
      nonwr := (!min,!max,Lazy.force sym) :: !nonwr
  in
  List.iter section obj.sections;
  int_to_buf data 0;
  strsym.value <- Int32.of_int (Buffer.length data);
  Buffer.add_buffer data strings;
  nonwrsym.value <- Int32.of_int (Buffer.length data);
  List.iter
    (fun (min,max,secsym) ->
      Reloc.abs !machine sect (Int32.of_int (Buffer.length data)) secsym;
      int_to_buf data (Int32.to_int min);
      Reloc.abs !machine sect (Int32.of_int (Buffer.length data)) secsym;
      int_to_buf data (Int32.to_int  max);
      int_to_buf data 0;
    )
    !nonwr;
  int_to_buf data 0;
  int_to_buf data 0;
  sect.data <- `String (Buffer.to_bytes data);
  obj.sections <- sect :: obj.sections;
  obj.symbols <-
    (Symbol.export sname sect 0l) ::
    strsym :: nonwrsym :: List.filter (fun x -> not (p x)) obj.symbols
    @ !syms;
  sname

(* Create a table for import symbols __imp_XXX *)

let add_import_table obj imports =
  let ptr_size = match !machine with `x86 -> 4 | `x64 -> 8 in
  let sect = Section.create ".imptbl" 0xc0300040l in
  obj.sections <- sect :: obj.sections;
  sect.data <- `String (Bytes.make (ptr_size * List.length imports) '\000');
  let i = ref 0 in
  List.iter
    (fun s ->
       let sym = Symbol.extern s in
       obj.symbols <-
         sym :: Symbol.export ("__imp_" ^ s) sect (Int32.of_int !i) ::
         obj.symbols;
       Reloc.abs !machine sect (Int32.of_int !i) sym;
       i := !i + ptr_size
    )
    imports


(* Create a table that lists exported symbols (adress,name) *)

let add_export_table obj exports symname =
  let sect = Section.create ".exptbl" 0xc0300040l in
  let data = Buffer.create 1024 in
  let strings = Buffer.create 1024 in
  let strsym = Symbol.intern sect 0l in
  obj.symbols <- strsym :: (Symbol.export symname sect 0l) :: obj.symbols;
  let exports = List.sort Pervasives.compare exports in
  (* The runtime library assumes that names are sorted! *)
  int_to_buf data (List.length exports);
  List.iter
    (fun s  ->
       let sym = Symbol.extern s in
       obj.symbols <- sym :: obj.symbols;
       Reloc.abs !machine sect (Int32.of_int (Buffer.length data)) sym;
       int_to_buf data 0;

       Reloc.abs !machine sect (Int32.of_int (Buffer.length data)) strsym;
       int_to_buf data (Buffer.length strings);
       Buffer.add_string strings (drop_underscore obj s);
       Buffer.add_char strings '\000';
    )
    exports;
  strsym.value <- Int32.of_int (Buffer.length data);
  let s = Bytes.cat (Buffer.to_bytes data) (Buffer.to_bytes strings) in
  sect.data <- `String s;
  obj.sections <- sect :: obj.sections

(* A master relocation table points to all the relocation tables
   in the DLL *)

let add_master_reloc_table obj names symname =
  let sect = Section.create ".mreltbl" 0xc0300040l in
  let data = Buffer.create 1024 in
  obj.symbols <- (Symbol.export symname sect 0l) :: obj.symbols;
  List.iter
    (fun s  ->
       let sym = Symbol.extern s in
       obj.symbols <- sym :: obj.symbols;
       Reloc.abs !machine sect (Int32.of_int (Buffer.length data)) sym;
       int_to_buf data 0;
    )
    names;
  int_to_buf data 0;
  sect.data <- `String (Buffer.to_bytes data);
  obj.sections <- sect :: obj.sections



let collect_dllexports obj =
  let dirs = Coff.directives obj in
  let l =
    List.map
      (function
        | (_,x::_) -> x
        | _ -> assert false
      )
      (List.find_all (fun (cmd,_args) -> String.uppercase_ascii cmd = "EXPORT") dirs)
  in
  match !toolchain with
  | `MSVC | `MSVC64 -> List.map (drop_underscore obj) l
  | _ -> l

let collect f l =
  List.fold_left
    (fun accu x -> match f x with None -> accu | Some y -> y :: accu)
    []
    l

let cmd_verbose cmd =
  if !verbose >= 1 then Printf.printf "+ %s\n" cmd;
  Sys.command cmd


let parse_dll_exports fn =
  let ic = open_in fn in
  let exps = ref [] in
  try
    while input_line ic <> "[Ordinal/Name Pointer] Table" do () done;
    while true do
      let s = input_line ic in
      let r = String.index s ']' in
      let sym = String.sub s (r+2) (String.length s - r - 2) in
      exps := ("_" ^ sym,0) :: !exps;
    done;
    assert false
  with Not_found | End_of_file ->
    close_in ic;
    !exps


let dll_exports fn = match !toolchain with
  | `MSVC | `MSVC64 | `LIGHTLD ->
      failwith "Creation of import library not supported for this toolchain"
  | `GNAT | `CYGWIN | `CYGWIN64 | `MINGW | `MINGW64 ->
      let dmp = temp_file "dyndll" ".dmp" in
      if cmd_verbose (Printf.sprintf "%s -p %s > %s" !objdump fn dmp) <> 0
      then failwith "Error while extracting exports from a DLL";
      parse_dll_exports dmp


let patch_output filename =
  match !stack_reserve with
  | Some x ->
      let filename =
        if not (Sys.file_exists filename) && (Sys.file_exists (filename ^ ".exe")) then filename ^ ".exe"
        else filename
      in
      let filename =
        if !use_cygpath then cygpath1 filename
        else filename
      in
      begin try Stacksize.set_stack_reserve filename x
      with exn ->
        Printf.eprintf "Cannot set stack reserve: %s"
          (Printexc.to_string exn)
      end
  | None -> ()


(* Extract the set of external symbols required by an object. *)
(* If the object requires "__imp_X", and "X" is available in one of the objects/libraries
   (but not "__imp_X" itself), then we consider that "X" is required.
   Indeed, we will create "__imp_X" (with a redirection to "X").
   Collect such cases in "imported".
*)
let needed imported defined unalias obj =
  let rec normalize name =
    try
      let r = unalias name in
      if r <> name then normalize r else r
    with Not_found -> name
  in
  let normalize_imp name =
    match check_prefix "__imp_" name with
    | Some s when not (StrSet.mem name defined) ->
        imported := StrSet.add s !imported;
        if StrSet.mem s defined then s else name
    | _ -> name
  in
  List.fold_left
    (fun accu sym ->
       if Symbol.is_extern sym then StrSet.add (normalize_imp (normalize sym.sym_name)) accu
       else accu
    )
    StrSet.empty
    obj.symbols

let build_dll link_exe output_file files exts extra_args =
  let main_pgm = link_exe <> `DLL in

  (* fully resolve filenames, eliminate duplicates *)
  let _, files =
    List.fold_left
      (fun (seen, accu) fn ->
         let fn = find_file fn in
         let k = String.lowercase_ascii fn in
         if StrSet.mem k seen then (seen, accu)
         else (StrSet.add k seen, fn :: accu)
      ) (StrSet.empty, []) files in
  let files = List.rev files in

  (* load given files *)
  let loaded_filenames : (string,unit) Hashtbl.t = Hashtbl.create 16 in
  let read_file fn =
    if Lib.is_dll fn then `Lib ([], dll_exports fn)
    else Lib.read fn
  in
  let files = List.map (fun fn -> fn, read_file fn) files in

  List.iter (fun (fn,_) -> Hashtbl.add loaded_filenames fn ()) files;

  let objs = collect (function (f, `Obj x) -> Some (f,x) | _ -> None) files in
  let libs = collect (function (f, `Lib (x,_)) -> Some (f,x) | _ -> None) files in

  (* Collect all the available symbols, including those defined
     in default libraries *)
  let defined, from_imports, unalias =
    let aliases = Hashtbl.create 16 in
    let defined = ref StrSet.empty in
    let from_imports = ref StrSet.empty in (* symbols from import libraries *)
    let add_def s = defined := StrSet.add s !defined in

    let collected = Hashtbl.create 8 in
    let rec collect_defined_obj obj =
      (* see comments on Cygwin64 COMDATA sections.  Here we give a
         unique name to the internal symbol.  We use ?? to ensure the
         symbol is not exported in flexdll export table (see
         exportable function) *)
      List.iter
        (fun sym ->
           if has_prefix ".refptr." sym.sym_name then
             sym.sym_name <- Printf.sprintf "??flexrefptr%i" (Oo.id (object end))
        )
        obj.symbols;

      (* Collect aliases *)
      List.iter
        (fun (x, y) ->
           if !verbose >= 2 then Printf.printf "alias %s -> %s\n" x y;
           Hashtbl.add aliases x y
        )
        (Coff.aliases obj);


      (* Iterates through DEFAULTLIB directives *)
      let register_deflib fn =
        if not (is_crt_lib fn) then
          let fn = find_file fn in
          if not (Hashtbl.mem loaded_filenames fn)
          then (Hashtbl.add loaded_filenames fn (); collect_file fn)
      in
      if not !builtin_linker && !use_default_libs && not !custom_crt then
        List.iter
          (fun (cmd, args) ->
             if String.uppercase_ascii cmd = "DEFAULTLIB" then List.iter register_deflib args
          )
          (Coff.directives obj);

      (* Collect defined symbols *)
      List.iter
        (fun sym -> if Symbol.is_defin sym then add_def sym.sym_name)
        obj.symbols

    and collect_file fn =
      if not (Hashtbl.mem collected (String.lowercase_ascii fn)) then begin
        Hashtbl.replace collected (String.lowercase_ascii fn) ();
        if !verbose >= 2 then Printf.printf "** open: %s\n" fn;
        collect_defined fn (Lib.read fn)
      end

    and collect_defined fn = function
      | `Obj obj -> collect_defined_obj obj
      | `Lib (objs,imports) ->
          List.iter (fun (_, obj) -> collect_defined_obj obj) objs;
          List.iter
            (fun (s,_) ->
               if !verbose >= 2 then
                 Printf.printf "lib %s import symbol %s\n%!" fn s;
               from_imports := StrSet.add s !from_imports;
               add_def s;
               add_def ("__imp_" ^ s)
            )
            imports
    in
    List.iter
      (fun (fn,x) ->
         Hashtbl.replace collected (String.lowercase_ascii fn) ();
         collect_defined fn x
      )
      files;
    if !use_default_libs then List.iter (fun fn -> collect_file (find_file fn)) !default_libs;
    List.iter (fun fn -> collect_file (find_file fn)) exts;

    if main_pgm then add_def (usym "static_symtable")
    else add_def (usym "reloctbl");

    if !machine = `x64 then add_def "__ImageBase"
    else add_def "___ImageBase";

    !defined, !from_imports, (Hashtbl.find aliases)
  in

  (* Determine which objects from the given libraries should be linked
     in. First step: find the mapping (symbol -> object) for these
     objects. *)
  let defined_in =
    let defined_in = Hashtbl.create 16 in
    let def_in_obj fn (objname, obj) =
      List.iter
        (fun sym ->
           if Symbol.is_defin sym
           then begin
             if !explain then
               Printf.printf "Symbol %s found in %s(%s)\n%!" sym.sym_name fn
                 objname;
             Hashtbl.replace defined_in sym.sym_name (fn, objname, obj);
           end
        )
        obj.symbols
    in
    List.iter
      (fun (fn,objs) ->
         if !explain then Printf.printf "Scanning lib %s\n%!" fn;
         List.iter (def_in_obj fn) objs
      )
      libs;
    Hashtbl.find defined_in
  in

  let imported_from_implib = ref StrSet.empty in
  let imported = ref StrSet.empty in

  let imports obj =
    let n = needed imported defined unalias obj in
    imported_from_implib := StrSet.union !imported_from_implib (StrSet.inter n from_imports);
    let undefs = StrSet.diff n defined in
    StrSet.filter
      (fun s ->
         match check_prefix "__imp_" s with
         | Some _ -> false
         | None -> s <> "environ"  (* special hack for Cygwin64 *)
      )
      undefs
  in

  (* Second step: transitive closure, starting from given objects *)

  let libobjects  = Hashtbl.create 16 in
  let reloctbls = ref [] in
  let exported = ref StrSet.empty in

  List.iter (fun s -> exported := StrSet.add (usym s) !exported) !defexports;

  let record_obj obj =
    if !builtin_linker then ""
    else begin
      let fn = temp_file "dyndll" (ext_obj ()) in
      let oc = open_out_bin fn in
      Coff.put oc obj;
      close_out oc;
      fn
    end
  in

  let add_reloc name obj imps =
    if !show_imports && not (StrSet.is_empty imps) then (
      Printf.printf "** Imported symbols for %s:\n" name;
      StrSet.iter print_endline imps
    );
    let sym = add_reloc_table obj name (fun s -> StrSet.mem s.sym_name imps) in
    reloctbls := sym :: !reloctbls
  in

  let errors = ref false in
  let error_imports name imps =
    if main_pgm then begin
      Printf.eprintf "** Cannot resolve symbols for %s:\n %s\n%!"
        name
        (String.concat "\n " (StrSet.elements imps));
      errors := true
    end
  in

  let close_obj name imps obj =
    error_imports name imps;
    add_reloc name obj imps;
    record_obj obj
  in

  let dll_exports = ref StrSet.empty in
  let rec link_obj fn obj =
    List.iter
      (fun sym ->
         if Symbol.is_defin sym && exportable sym.sym_name
         then exported := StrSet.add sym.sym_name !exported
      )
      obj.symbols;

    dll_exports := List.fold_left (fun accu x -> StrSet.add x accu)
        !dll_exports (collect_dllexports obj);
    StrSet.iter
      (fun s ->
        if StrSet.mem s !exported then ()
        else
          try
            let (libname, objname, _) as o = defined_in s in
            if !explain then
              Printf.printf "%s -> %s(%s) because of %s\n%!" fn libname objname s;
            link_libobj o
          with Not_found ->
            if !explain then
              Printf.printf "%s needs %s (not found)\n%!" fn s
      )
      (needed imported defined unalias obj)

  and link_libobj (libname,objname,obj) =
    if Hashtbl.mem libobjects (libname,objname) then ()
    else (Hashtbl.replace libobjects (libname,objname) (obj,imports obj);
          link_obj (Printf.sprintf "%s(%s)" libname objname) obj)
  in

  let redirect = Hashtbl.create 16 in
  List.iter
    (fun (fn, obj) ->
       link_obj fn obj;
       let imps = imports obj in
       if StrSet.is_empty imps then ()
       else Hashtbl.replace redirect fn (close_obj fn imps obj);
    ) objs;

  let need_lib = Hashtbl.create 16 in
  Hashtbl.iter
    (fun (libname,objname) (obj,imps) ->
      if StrSet.is_empty imps
      then Hashtbl.replace need_lib libname ()
          (* the linker will find this object in this library *)
      else begin
        if !explain then
          Printf.printf "Library object %s(%s) needs to be rewritten\n"
            libname objname;
        Hashtbl.add redirect libname
          (close_obj (Printf.sprintf "%s(%s)" libname objname) imps obj)
      end
    )
    libobjects;

  if !show_exports then (
    Printf.printf "** Exported symbols:\n";
    StrSet.iter print_endline !exported;
    Printf.printf "** Symbols from import libs:\n";
    StrSet.iter print_endline !imported_from_implib;
  );

  if !reexport_from_implibs then
    exported := StrSet.union !exported !imported_from_implib;

  (* Create the descriptor object *)
  let obj = Coff.create !machine in

  if not (StrSet.is_empty !imported) then begin
(*
    Printf.printf "** __imp symbols:\n";
    StrSet.iter print_endline !imported;
*)
    add_import_table obj (StrSet.elements !imported);
    let undef_imports = StrSet.diff !imported defined in
    if not (StrSet.is_empty undef_imports) then begin
      error_imports "descriptor object" undef_imports;
      add_reloc "descriptor object" obj undef_imports;
    end
  end;

  add_export_table obj (if !noexport then [] else StrSet.elements !exported)
    (usym (if main_pgm then "static_symtable" else "symtbl"));
  if not main_pgm then add_master_reloc_table obj !reloctbls (usym "reloctbl");

  if !errors then
    exit 2;

  if !builtin_linker then begin
    let objs = List.map
        (function
           | (_, `Obj obj) -> obj
           | (fn, _) -> failwith ("File is not an object file: " ^ fn)
        ) files
    in
    let oc = open_out_bin output_file in
    Create_dll.create_dll oc (List.rev (obj :: objs));
    close_out oc
  end else

  let descr = record_obj obj in
  let files =
    List.flatten
      (List.map
         (fun (fn,d) ->
           let all = Hashtbl.find_all redirect fn in
           if all = [] then [fn]
           else
             match d with
             | `Lib _ when Hashtbl.mem need_lib fn -> all @ [fn]
             | `Lib (_, []) | `Obj _ -> all
             | `Lib _ -> all @ [fn]
            (* Note: extracted object have higher priorities
               than objects embedded in the library, so this is ok.
               We always keep libraries with import symbols.
               For mingw, it is necessary to put the library after
               extracted objects. *)
         )
         files
      )
    @ exts in

  let cmdline = new_cmdline () in
  let files = quote_files cmdline files in
  let descr = Filename.quote descr in

  begin
    match !deffile with
    | Some x when not !dry_mode ->
        let fn =
          if x = "" then Filename.chop_extension output_file ^ ".def"
          else x
        in
        if !verbose >= 1 then Printf.printf "Generate %s\n%!" fn;
        let oc = open_out fn in
        Printf.fprintf oc "LIBRARY %s\n" output_file;
        Printf.fprintf oc "EXPORTS\n";
        StrSet.iter (Printf.fprintf oc "  %s\n") !dll_exports;
        close_out oc
    | _ -> ()
  end;

  let cmd = match !toolchain with
    | `MSVC | `MSVC64 ->
        (* Putting the file the descriptor object at the beginning
           with MSVC compilers seems to break Stack overflow recovery
           in OCaml. No idea why. *)

        let implib =
          if !implib then
            Filename.chop_extension output_file ^ ".lib"
          else
            temp_file "dyndll_implib" ".lib"
        in
        (* VS 2017.3 doesn't seem to be able to cope with /implib: existing but
           being an empty file. *)
        let c = open_out implib in output_string c "x"; close_out c;
        let _impexp = add_temp (Filename.chop_suffix implib ".lib" ^ ".exp") in
        let extra_args =
          if !custom_crt then "/nodefaultlib:LIBCMT /nodefaultlib:MSVCRT " ^ extra_args
          else "msvcrt.lib " ^ extra_args
        in

        let extra_args =
          if !machine = `x64 then (Printf.sprintf "/base:%s " !base_addr) ^ extra_args else extra_args
        in
        (* Flexdll requires that all images (main programs and all the DLLs) are
           not too far away. This is needed because of the 32-bit relative relocations
           (e.g. function calls). It seems that passing such a /base argument to link.exe
           gives some hope that this will be the case. Problems observed otherwise
           with the Windows 7 SDK in 64-bit mode. *)

        Printf.sprintf
          "link /nologo %s%s%s%s%s /implib:%s /out:%s /subsystem:%s %s %s %s"
          (if !verbose >= 2 then "/verbose " else "")
          (if link_exe = `EXE then "" else "/dll ")
          (if main_pgm then "" else "/export:symtbl /export:reloctbl ")
          (if main_pgm then "" else if !noentry then "/noentry " else
          let s =
            match !machine with
            | `x86 -> "FlexDLLiniter@12"
            | `x64 -> "FlexDLLiniter"
          in
          Printf.sprintf "/entry:%s " s
          )
          (mk_dirs_opt "/libpath:")
          (Filename.quote implib)
          (Filename.quote output_file)
          !subsystem
          files descr
          extra_args
    | `CYGWIN | `CYGWIN64 ->
        let def_file =
          if main_pgm then ""
          else
            let def_file, oc = open_temp_file "flexlink" ".def" in
            Printf.fprintf oc "EXPORTS\n  reloctbl\n  symtbl\n";
            close_out oc;
            Filename.quote def_file
        in
        let extra_args =
          if !machine = `x64 then "-Xlinker --image-base -Xlinker 0x10000 " ^ extra_args else extra_args
        in
        Printf.sprintf
          "%s %s%s -L. %s %s -o %s %s %s %s %s"
          !gcc
          (if link_exe = `EXE then "" else "-shared ")
          (if main_pgm then "" else if !noentry then "-Wl,-e0 " else if !machine = `x86 then "-Wl,-e_FlexDLLiniter@12 " else "-Wl,-eFlexDLLiniter ")
          (mk_dirs_opt "-I")
          (mk_dirs_opt "-L")
          (Filename.quote output_file)
          descr
          files
          def_file
          extra_args
    | `MINGW | `MINGW64 | `GNAT ->
        let def_file =
          if main_pgm then ""
          else
            let def_file, oc = open_temp_file "flexlink" ".def" in
            Printf.fprintf oc "EXPORTS\n  reloctbl\n  symtbl\n";
            close_out oc;
            Filename.quote def_file
        in
        Printf.sprintf
          "%s -m%s %s%s -L. %s %s -o %s %s %s %s %s %s"
          !gcc
          !subsystem
          (if link_exe = `EXE then "" else "-shared ")
          (if main_pgm then "" else if !noentry then "-Wl,-e0 " else if !machine = `x86 then "-Wl,-e_FlexDLLiniter@12 " else "-Wl,-eFlexDLLiniter ")
          (mk_dirs_opt "-I")
          (mk_dirs_opt "-L")
          (Filename.quote output_file)
          descr
          files
          def_file
          (if !implib then "-Wl,--out-implib=" ^ Filename.quote (Filename.chop_extension output_file ^ ".a") else "")
          extra_args
    | `LIGHTLD ->
        no_merge_manifest := true;
        Printf.sprintf
          "ld %s%s -o %s %s %s %s %s"
          (if link_exe = `EXE then "" else "--shared ")
          (if main_pgm then "" else if !noentry then "-e0 " else "-e FlexDLLiniter@12 ")
          (Filename.quote output_file)
          descr
          files
          (if !implib then "--out-implib " ^ Filename.quote (Filename.chop_extension output_file ^ ".a") else "")
          extra_args
  in
  if !verbose >= 1 || !dry_mode then Printf.printf "+ %s\n%!" cmd;
  if not !dry_mode then begin
    let manifest_file = output_file ^ ".manifest" in
    safe_remove manifest_file;
    run_command cmdline cmd;

    if (not !no_merge_manifest) && !merge_manifest && (not !real_manifest || Sys.file_exists manifest_file)
    then begin
      let fn =
        if !real_manifest then manifest_file
        else
          let default_manifest =
            match !machine with
            | `x86 -> "default.manifest"
            | `x64 -> "default_amd64.manifest"
          in
          Filename.concat flexdir default_manifest
      in
      let mcmd =
        Printf.sprintf "mt -nologo -outputresource:%s -manifest %s"
          (Filename.quote (if link_exe = `EXE then output_file
                           else output_file ^ ";#2"))
          (Filename.quote fn)
      in
      if !verbose >= 1 then Printf.printf "+ %s\n%!" mcmd;
      if Sys.command mcmd <> 0 then
        failwith "Error while merging the manifest";
      safe_remove manifest_file;
    end;
    patch_output output_file
 end

let ends_with s suf =
  let rec aux s suf suflen offset i =
    i >= suflen || (s.[i + offset] = suf.[i]
                   && aux s suf suflen offset (i+1)) in
  let slen = String.length s in
  let suflen = String.length suf in
  slen >= suflen && aux s suf suflen (slen - suflen) 0

let strip s =
  let rec search i s =
    if s.[i] = ' ' then search (i+1) s
    else String.sub s i (String.length s - i)
  in
  search 0 s

let read_gnatls () =
   (* This function is used by the GNAT toolchain to compute the include
      directory. gnatls actually returns with an error code different to 0, so
      we need to accept the error here. *)
   let str_l = get_output ~accept_error:true ("gnatls -v") in
   let ada_include =
     List.hd (List.filter (fun s -> ends_with s "adainclude") str_l) in
   Filename.dirname (strip ada_include)

let split str sep =
  let p = String.index str sep in
  let slen = String.length str in
  String.sub str 0 p, String.sub str (p + 1) (slen - p - 1)

let nsplit str sep =
  if str = "" then []
  else
    let rec loop acc pos =
      if pos > String.length str then
        List.rev acc
      else
        let i = try String.index_from str pos sep with Not_found -> String.length str in
        loop (String.sub str pos (i - pos) :: acc) (i + 1)
    in
    loop [] 0

let normalize_path path =
  let path = nsplit path '/' in
  let rec loop acc path =
    match path with
    | "."  :: path -> loop acc path
    | ".." :: path -> loop (List.tl acc) path
    | elem :: path -> loop (elem :: acc) path
    | [] -> List.rev acc
  in
  let path = loop [] path in
  String.concat "/" path

let remove_duplicate_paths paths =
  let set = Hashtbl.create 16 in
  let rec loop paths =
    match paths with
    | path :: paths ->
        begin try
          Hashtbl.find set path;
          loop paths
        with Not_found ->
          Hashtbl.add set path ();
          path :: loop paths
        end
    | [] -> []
  in
  loop paths

let setup_toolchain () =
  let mingw_libs pre =
    gcc := pre ^ "gcc";
    objdump := pre ^ "objdump";
    let rec get_lib_search_dirs input =
      match input with
      | entry :: input ->
          begin try
            match split entry '=' with
            | "libraries: ", paths -> nsplit paths ':'
            | _ -> get_lib_search_dirs input
          with Not_found ->
            get_lib_search_dirs input
          end
      | [] -> []
    in
    let lib_search_dirs =
      get_output (!gcc ^ " -print-search-dirs")
      |> get_lib_search_dirs
      |> List.map normalize_path
      |> remove_duplicate_paths
    in
    search_path := !dirs @ lib_search_dirs;
    if !verbose >= 1  then begin
      print_endline "lib search dirs:";
      List.iter (Printf.printf "  %s\n%!") lib_search_dirs;
    end;
    default_libs :=
      ["-lmingw32"; "-lgcc"; "-lmoldname"; "-lmingwex"; "-lmsvcrt";
       "-luser32"; "-lkernel32"; "-ladvapi32"; "-lshell32" ];
    if !exe_mode = `EXE then default_libs := "crt2.o" :: !default_libs
    else default_libs := "dllcrt2.o" :: !default_libs
  in
  match !toolchain with
  | _ when !builtin_linker ->
      search_path := !dirs;
      add_flexdll_obj := false;
      noentry := true
  | `CYGWIN | `CYGWIN64 ->
      gcc := "gcc";
      objdump := "objdump";
      search_path :=
        !dirs @
          [
           "/lib";
           "/lib/w32api";
           Filename.dirname (get_output1 ~use_bash:true "gcc -print-libgcc-file-name");
          ];
      default_libs := ["-lkernel32"; "-luser32"; "-ladvapi32";
                       "-lshell32"; "-lcygwin"; "-lgcc"]
  | `MSVC | `MSVC64 ->
      search_path := !dirs @
        parse_libpath (try Sys.getenv "LIB" with Not_found -> "");
      if not !custom_crt then
        default_libs := ["msvcrt.lib"]
  | `MINGW ->
      mingw_libs Version.mingw_prefix
  | `MINGW64 ->
      mingw_libs Version.mingw64_prefix
  | `GNAT ->
   (* This is a plain copy of the mingw version, but we do not change the
      prefix and use "gnatls" to compute the include dir. *)
    search_path :=
      !dirs @
      [
       Filename.dirname (get_output1 (!gcc ^ " -print-libgcc-file-name"));
       read_gnatls ();
      ];
    default_libs :=
      ["-lmingw32"; "-lgcc"; "-lmoldname"; "-lmingwex"; "-lmsvcrt";
       "-luser32"; "-lkernel32"; "-ladvapi32"; "-lshell32" ];
    if !exe_mode = `EXE then default_libs := "crt2.o" :: !default_libs
    else default_libs := "dllcrt2.o" :: !default_libs
  | `LIGHTLD ->
      search_path := !dirs

let display_msvc_output file name =
  let c = open_in file in
  try
    let first = input_line c in
    if first <> (Filename.basename name) then
      print_string first;
    while true do
      print_string (input_line c)
    done
  with _ ->
    close_in c

let compile_if_needed file =
  if Filename.check_suffix file ".c" then begin
    let tmp_obj = temp_file "dyndll" (ext_obj ()) in
    let (pipe, stdout) =
      if (!toolchain = `MSVC || !toolchain = `MSVC64) && !verbose < 2 && not !dry_mode then
        try
          let (t, c) = open_temp_file "msvc" "stdout" in
          close_out c;
          (Printf.sprintf " > %s" (Filename.quote t), t)
        with _ ->
          ("", "")
      else
        ("", "") in
    let cmd = match !toolchain with
      | `MSVC | `MSVC64 ->
          Printf.sprintf
            "cl /c /MD /nologo /Fo%s %s %s%s"
            (Filename.quote tmp_obj)
            (mk_dirs_opt "/I")
            file
            pipe
      | `CYGWIN | `CYGWIN64 ->
          Printf.sprintf
            "gcc -c -o %s %s %s"
            (Filename.quote tmp_obj)
            (mk_dirs_opt "-I")
            file
      | `MINGW | `MINGW64 | `GNAT ->
          Printf.sprintf
            "%s -c -o %s %s %s"
            !gcc
            (Filename.quote tmp_obj)
            (mk_dirs_opt "-I")
            (Filename.quote file)
      | `LIGHTLD ->
          failwith "Compilation of C code is not supported for this toolchain"
    in
    if !verbose >= 1 || !dry_mode then Printf.printf "+ %s\n%!" cmd;
    let exit = if !dry_mode then 0 else Sys.command cmd in
    if pipe <> "" then display_msvc_output stdout file;
    if exit <> 0 then failwith "Error while compiling";
    tmp_obj
  end else
    file

let dump fn =
  let fn = find_file fn in
  Printf.printf "*** %s:\n" fn;
  match Lib.read fn with
  | `Lib (objs,imports) ->
      List.iter
        (fun (n,o) ->
          Printf.printf "** %s(%s):\n" fn n;
          Coff.dump o
        )
        objs;
      List.iter
        (fun (s,i) -> Printf.printf "** import: %s (%i)\n" s i)
        imports
  | `Obj o ->
      Coff.dump o

let all_files () =
  let files = List.rev (List.map compile_if_needed !files) in
  let f obj =
    let fn = Filename.concat flexdir obj in
    (* Allow the obj files to be stored in a different location *)
    if file_exists fn <> None then
      fn
    else
      obj in
  let tc = match !toolchain with
  | `MSVC -> "msvc.obj"
  | `MSVC64 -> "msvc64.obj"
  | `CYGWIN -> "cygwin.o"
  | `CYGWIN64 -> "cygwin64.o"
  | `MINGW64 -> "mingw64.o"
  | `GNAT    -> "gnat.o"
  | `MINGW | `LIGHTLD -> "mingw.o" in
  if !exe_mode <> `DLL then
    if !add_flexdll_obj then f ("flexdll_" ^ tc) :: files
    else files
  else
    if !noentry then files
    else f ("flexdll_initer_" ^ tc) :: files

let main () =
  parse_cmdline ();
  setup_toolchain ();

  use_cygpath :=
    begin
      match !toolchain, !cygpath_arg with
      | _, `Yes -> true
      | _, `No -> false
      | (`GNAT|`MINGW|`MINGW64|`CYGWIN|`CYGWIN64), `None ->
          begin match Sys.os_type with
          | "Unix" | "Cygwin" ->
              Sys.command "cygpath -S 2>/dev/null >/dev/null" = 0
          | "Win32" ->
              Sys.command "cygpath -S 2>NUL >NUL" = 0
          | _ -> assert false
          end
      | (`MSVC|`MSVC64|`LIGHTLD), `None -> false
    end;


  if !verbose >= 2 then (
    Printf.printf "** Use cygpath: %b\n" !use_cygpath;
    Printf.printf "** Search path:\n";
    List.iter print_endline !search_path;
    if !use_default_libs then begin
      Printf.printf "** Default libraries:\n";
      List.iter print_endline !default_libs;
    end
   );
  let files = all_files () in
  match !mode with
  | `DUMP -> List.iter dump files
  | `NORMAL ->
      build_dll !exe_mode !output_file files !exts
        (String.concat " " (List.map Filename.quote (List.rev !extra_args)))
  | `PATCH ->
      patch_output !output_file

let () =
  try main ()
  with
    | Failure s ->
        Printf.eprintf "** Fatal error: %s\n" s;
        exit 2
    | Invalid_argument s ->
        Printf.eprintf "** Fatal error: invalid argument (%s)\n" s;
        exit 2
    | Arg.Help s ->
        Printf.printf "%s\n%s\n" s footer;
        exit 0
    | Arg.Bad s ->
        Printf.eprintf "%s\n%s\n" s footer;
        exit 2
    | exn ->
        Printf.eprintf "** Error: %s\n" (Printexc.to_string exn);
        exit 2
