/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libepubgen project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef INCLUDED_LIBEPUBGEN_EPUBTEXTGENERATOR_H
#define INCLUDED_LIBEPUBGEN_EPUBTEXTGENERATOR_H

#include <librevenge/librevenge.h>
#include <librevenge-stream/librevenge-stream.h>

#include "libepubgen-api.h"
#include "libepubgen-decls.h"

namespace libepubgen
{

class EPUBPackage;

/** EPUB generator for text documents.
  */
class EPUBGENAPI EPUBTextGenerator : public librevenge::RVNGTextInterface
{
  // disable copying
  EPUBTextGenerator(const EPUBTextGenerator &);
  EPUBTextGenerator &operator=(const EPUBTextGenerator &);

  struct Impl;

public:
  explicit EPUBTextGenerator(EPUBPackage *package, EPUBSplitMethod split = EPUB_SPLIT_METHOD_PAGE_BREAK);
  EPUBTextGenerator(EPUBEmbeddingContact &contact, EPUBPackage *package);
  virtual ~EPUBTextGenerator();

  void setSplitHeadingLevel(unsigned level);
  void setSplitSize(unsigned size);

  /** Register a handler for embedded images.
    *
    * The handler must convert the image to one of the OPS Core Media Types.
    *
    * @param[in] mimeType the MIME type of the image
    * @param[in] imageHandler a function that converts the image to a
    *   suitable format
    *
    * @sa EPUBEmbeddedImage
    */
  void registerEmbeddedImageHandler(const librevenge::RVNGString &mimeType, EPUBEmbeddedImage imageHandler);

  /** Register a handler for embedded objects.
    *
    * @param[in] mimeType the MIME type of the object
    * @param[in] objectHandler a function that generates EPUB content
    *   from an object's data
    *
    * @sa EPUBEmbeddedObject
    */
  void registerEmbeddedObjectHandler(const librevenge::RVNGString &mimeType, EPUBEmbeddedObject objectHandler);

  virtual void setDocumentMetaData(const librevenge::RVNGPropertyList &propList);

  virtual void startDocument(const librevenge::RVNGPropertyList &propList);

  virtual void endDocument();

  virtual void definePageStyle(const librevenge::RVNGPropertyList &propList);

  virtual void defineEmbeddedFont(const librevenge::RVNGPropertyList &propList);

  virtual void openPageSpan(const librevenge::RVNGPropertyList &propList);
  virtual void closePageSpan();

  virtual void openHeader(const librevenge::RVNGPropertyList &propList);
  virtual void closeHeader();

  virtual void openFooter(const librevenge::RVNGPropertyList &propList);
  virtual void closeFooter();

  virtual void defineParagraphStyle(const librevenge::RVNGPropertyList &propList);

  virtual void openParagraph(const librevenge::RVNGPropertyList &propList);
  virtual void closeParagraph();

  virtual void defineCharacterStyle(const librevenge::RVNGPropertyList &propList);

  virtual void openSpan(const librevenge::RVNGPropertyList &propList);
  virtual void closeSpan();
  virtual void openLink(const librevenge::RVNGPropertyList &propList);
  virtual void closeLink();

  virtual void defineSectionStyle(const librevenge::RVNGPropertyList &propList);

  virtual void openSection(const librevenge::RVNGPropertyList &propList);
  virtual void closeSection();

  virtual void insertTab();
  virtual void insertSpace();
  virtual void insertText(const librevenge::RVNGString &text);
  virtual void insertLineBreak();

  virtual void insertField(const librevenge::RVNGPropertyList &propList);

  virtual void openOrderedListLevel(const librevenge::RVNGPropertyList &propList);
  virtual void openUnorderedListLevel(const librevenge::RVNGPropertyList &propList);
  virtual void closeOrderedListLevel();
  virtual void closeUnorderedListLevel();
  virtual void openListElement(const librevenge::RVNGPropertyList &propList);
  virtual void closeListElement();

  virtual void openFootnote(const librevenge::RVNGPropertyList &propList);
  virtual void closeFootnote();

  virtual void openEndnote(const librevenge::RVNGPropertyList &propList);
  virtual void closeEndnote();

  virtual void openComment(const librevenge::RVNGPropertyList &propList);
  virtual void closeComment();

  virtual void openTextBox(const librevenge::RVNGPropertyList &propList);
  virtual void closeTextBox();

  virtual void openTable(const librevenge::RVNGPropertyList &propList);
  virtual void openTableRow(const librevenge::RVNGPropertyList &propList);
  virtual void closeTableRow();
  virtual void openTableCell(const librevenge::RVNGPropertyList &propList);
  virtual void closeTableCell();
  virtual void insertCoveredTableCell(const librevenge::RVNGPropertyList &propList);
  virtual void closeTable();
  virtual void openFrame(const librevenge::RVNGPropertyList &propList);
  virtual void closeFrame();
  virtual void insertBinaryObject(const librevenge::RVNGPropertyList &propList);
  virtual void insertEquation(const librevenge::RVNGPropertyList &propList);

  virtual void openGroup(const librevenge::RVNGPropertyList &propList);
  virtual void closeGroup();

  virtual void defineGraphicStyle(const librevenge::RVNGPropertyList &propList);

  virtual void drawRectangle(const librevenge::RVNGPropertyList &propList);
  virtual void drawEllipse(const librevenge::RVNGPropertyList &propList);
  virtual void drawPolygon(const librevenge::RVNGPropertyList &propList);
  virtual void drawPolyline(const librevenge::RVNGPropertyList &propList);
  virtual void drawPath(const librevenge::RVNGPropertyList &propList);

  virtual void drawConnector(const librevenge::RVNGPropertyList &propList);

private:
  Impl *const m_impl;
};

} // namespace libepubgen

#endif // INCLUDED_LIBEPUBGEN_EPUBTEXTGENERATOR_H

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
